# SPDX-FileCopyrightText: 2022 RANCH Computing <contact@ranchcomputing.com>
# SPDX-License-Identifier: GPL-3.0-only

bl_info = {
    "name": "RANCHecker For Blender",
    "description": "Check your scene compatibility with the RANCH RenderFarm and prepare the archive to send on the RANCH",
    "author": "Ranch Computing",
    "version": (1,6,3),
    "blender": (2, 93, 0),
    "location": "View3D",
    "wiki_url": "https://doc.ranchcomputing.com/blender:start",
    "tracker_url": "https://www.ranchcomputing.com/support/help-center",
    "category": "RANCH Computing",
}

# bl_info must be at the very top
# pylint: disable=wrong-import-position
import hashlib
import json
import logging
import os
import pathlib
import platform
import re
import shutil
import socket
import ssl
import sys
import tempfile
import time
import urllib.request
import zipfile

from collections import OrderedDict

# pylint: disable=import-error
import bpy
import bpy.utils.previews

from bpy.props import (
    StringProperty,
    BoolProperty,
    IntProperty,
    PointerProperty,
    CollectionProperty,
    EnumProperty,
)

import addon_utils

global rfb_mod

# pylint: enable=wrong-import-position,import-error
# pylint: disable=missing-class-docstring,unused-private-member,unused-argument,too-many-nested-blocks

ssl._create_default_https_context = ssl._create_unverified_context

"""
        https://devtalk.blender.org/t/enumproperty-and-string-encoding/7835
        10 june 2019 - EnumProperty and string encoding:
            There is a known bug with using a callback,
            Python must keep a reference to the strings returned by the callback
            or Blender will misbehave or even crash.
"""
priority_items = []  #


def _add_priorityFromColl_cb(self, context):
    """
    Priority Enum item callback
    Fill enum items with priority collection in scene
    """

    global priority_items
    priority_items = []

    scene = context.scene

    for item in scene.coll_priority_items:
        identifier = str(item.priority_id)
        name = item.priority_str
        description = str(item.priority_id)

        priority_items.append((identifier, name, description))

    return priority_items


def get_desktop():
    try:
        if os.path.isdir(os.path.join(os.path.expanduser("~"), "Desktop")):
            desktop = os.path.join(os.path.expanduser("~"), "Desktop")
        else:
            desktop = os.path.expanduser("~")
    except:
        pass
    return desktop


class LogLine(bpy.types.PropertyGroup):
    """
    LogLine stores the properties of a log line:
        - icon (str): icon to use in the interface
        - msg (str): log message
    """

    icon: StringProperty(name="", default="")
    msg: StringProperty(name="", default="")


class RANCHeckerPreferences(bpy.types.AddonPreferences):
    """
    RANCHecker add_on preferences https://docs.blender.org/api/current/bpy.types.AddonPreferences.html

    Properties:
        - verbose_logging (bool): Activate verbose debugging
        - log_lines (CollectionProperty): Log lines
        - log_line_active (IntProperty): Index of the current log line
    """

    # this must match the add-on name,
    # use '__package__' when defining this in a submodule of a python package.
    # use '__name__' in case of a monofile addon module.
    # https://docs.blender.org/api/current/bpy.types.AddonPreferences.html
    bl_idname = __package__

    verbose_logging: BoolProperty(
        name="Activate verbose logging", description="for more messages in the interface"
    )

    use_blender_asset_tracer: BoolProperty(
        name="Use BAT (Blender Asset Tracer)", description="Use BAT for packing", default=False
    )

    compression_list = [
        ("0", "No Compression", "lvl 0, fast creation, slow upload, \nrecommended with +50mbps network speed"),
        ("3", "Balanced", "lvl 3, average creation and upload \nrecommended for 10 to 50mbps network speed"),
        ("6", "Max", "lvl 6, slow creation, fast upload() \nrecommended with -10mbps network speed"),
    ]

    vub_compression_level: EnumProperty(
        name="Archive Compression",
        description="Choose compression level according to your upload speed and disk space, \nnot compatible with BAT and RenderMan Packer",
        items=compression_list,
        default="3",
    )

    use_tex_rman_packer: BoolProperty(
        name="Use TEX (RenderMan)",
        description="RenderMan Packer will collect and use .tex file in texture paths, \n for RenderMan 26.3 and above only",
        default=False,
    )

    # storage properties (to persist on scene reload, but not shown in the preference panel)
    log_lines: CollectionProperty(type=LogLine)
    log_line_active: IntProperty(default=0)

    @classmethod
    def singleton(cls, context=None) -> "RANCHeckerPreferences":
        if not context:
            context = bpy.context
        return context.preferences.addons[cls.bl_idname].preferences

    def draw(self, context):
        self.layout.use_property_split = True
        self.layout.use_property_decorate = False
        layout = self.layout
        main_box = layout.box().column(align=True)
        main_box.label(text="Main Settings")

        main_box.prop(self, "vub_compression_level")
        main_box.prop(self, "verbose_logging")
        main_box.prop(self, "use_blender_asset_tracer")

        rdm_box = layout.box().column(align=True)
        rdm_box.label(text="RenderMan Settings")

        rdm_box.prop(self, "use_tex_rman_packer")

    # @classmethod
    # def register(cls):
    #     bpy.types.Scene.vub_compression = bpy.props.CollectionProperty(type=)

    # @classmethod
    # def unregister(cls):
    #     del bpy.types.Scene.vub_compression

    # populate: bpy.props.EnumProperty(name="", default="BALANCED")


class SettingsProp(bpy.types.PropertyGroup):
    destination_bool: BoolProperty(name="Destination", description="Saving archive directory")
    str_destination: StringProperty(name="", default=get_desktop(), subtype="DIR_PATH")
    archive_path: StringProperty(name="archive path", default="")
    message: StringProperty(name="dialog message", default="")
    test_frames_bool: BoolProperty(
        name="Test frames",
        description=(
            "Upload project on start middle and end frames\n"
            "Then reuse from your dashboard without specifying"
            " frames for full frame range"
        ),
    )
    skip_email_bool: BoolProperty(name="Skip email", description="Do not send project emails")

    priority: EnumProperty(
        name="priority",
        description="Priority to use on the RANCH",
        items=_add_priorityFromColl_cb,
    )

    @classmethod
    def register(cls):
        bpy.types.Scene.ranch_settings = PointerProperty(type=cls)

    @classmethod
    def unregister(cls):
        del bpy.types.Scene.ranch_settings


class ItemPriority(bpy.types.PropertyGroup):
    @classmethod
    def register(cls):
        bpy.types.Scene.coll_priority_items = bpy.props.CollectionProperty(type=ItemPriority)

    @classmethod
    def unregister(cls):
        del bpy.types.Scene.coll_priority_items

    priority_str: bpy.props.StringProperty(name="priority_str", default="")
    priority_id: bpy.props.IntProperty(name="priority_id", default=0)


# splitpath supports "/" and "\"
def splitpath(s):
    i = max(s.rfind("/"), s.rfind("\\"))
    if i < 0:  # no separator found
        return "", s
    if i == 0:  # separator at the very beginning
        return s[: i + 1], s[i + 1 :]
    if i == 1 and s[0] == "/":  # blender special case: // prefix
        return s[: i + 1], s[i + 1 :]

    return s[:i], s[i + 1 :]


def md5_basename(folder):
    """
    Compute a folder name (basename + md5 of the full path).
    Any update to this function must be copied to RANCHSet/RANCHecker.
    """
    # find part of path without "."
    parents, basename, i = folder, "", 0
    while not basename or "." in basename or ":" in basename:
        previous_parents = parents
        parents, basename = splitpath(parents)
        if not basename and parents == previous_parents:
            break
        if i > 1000:  # prevent infinite loop
            print("infinite splitpath on", folder)
            raise Exception("infinite splitpath on: " + folder)
        i += 1

    # Replace / with \ to have the same behavior on windows and linux/unix platforms
    if folder.startswith("//"):
        folder = "//" + folder[2:].replace("/", "\\")
    else:
        folder = folder.replace("/", "\\")
    return basename + "_" + hashlib.md5(folder.encode()).hexdigest()


class Utils:
    forbidden_chars = re.compile("[^a-zA-Z0-9-_]+")
    OUTPUT_NAME_FORBIDDEN_CHARS = re.compile(r"[^a-zA-Z0-9_.-]+")
    RENDERMAN_OUTPUT_FORMATS = ("exr", "png", "tga", "tif")  # Default exr

    _digit_sequence = re.compile(r"\d{1,}")  # matches sequence of 1 digit or more

    def __init__(self):
        pass

    @staticmethod
    def include_files_matching(pattern: str):
        """ "
        include_files_matching will return an ignore_function,
        which will ignore all files not matching the pattern.
        """
        compiled = re.compile(pattern)

        return lambda root, fileNames: [f for f in fileNames if not compiled.match(f)]

    @classmethod
    def folder_and_ignore_function_for_sequence(cls, filepath, is_sequence=False):
        """
        Return
            - The source folder of a file or a sequence of files
            - The function ignoring the files not matching a pattern
              (Ex: single file name, frame sequence for caches or images sequence, udim ...etc )
        """
        folder, filename = splitpath(filepath)

        if not is_sequence:
            return folder, lambda root, fileNames: [f for f in fileNames if f != filename]

        filename = re.escape(filename)
        pattern = cls._digit_sequence.sub(
            lambda _: r"\d+",  # lambda needed to prevent repl from being escaped
            filename,
        )
        return folder, cls.include_files_matching(pattern)

    @staticmethod
    def additional_light_folders(light, log):
        try:
            for l_node in light.node_tree.nodes:
                # IES texture external file (not packed)
                if l_node.type == "TEX_IES" and l_node.mode == "EXTERNAL":
                    yield Utils.folder_and_ignore_function_for_sequence(l_node.filepath)
                elif l_node.type == "TEX_IMAGE" and l_node.image.source == "MOVIE":
                    _, img_ext = os.path.splitext(bpy.path.abspath(l_node.image.filepath))
                    if img_ext.upper() == ".MP4":
                        yield Utils.folder_and_ignore_function_for_sequence(l_node.image.filepath)
                    else:
                        log.warning("Video texture format no supported: " + l_node.image.filepath)

        except AttributeError:
            pass

    @staticmethod
    def subscene_folders(subscene, log):
        log.info("Linked scene found: " + subscene.name)
        yield Utils.folder_and_ignore_function_for_sequence(subscene.filepath)

    @staticmethod
    def additional_folders(log):
        # External files not packed:
        # Search for IES, cache files, vdb, textures: images sequences, .mp4 or udim
        # Return dirname, function ignoring files not needed

        # search subscene libraries
        for subscene in bpy.data.libraries:
            yield from Utils.subscene_folders(subscene, log)
        #     yield Utils.folder_and_ignore_function_for_sequence(subscene, log)

        # search IES or MP4 in light texture
        for light in bpy.data.lights:
            yield from Utils.additional_light_folders(light, log)

        for cache in bpy.data.cache_files:  # ABC ou USD
            yield Utils.folder_and_ignore_function_for_sequence(cache.filepath, cache.is_sequence)

        # bpy.ops.file.pack_all() does not pack vdb sequence, only the 1st is packed
        for volume in bpy.data.volumes:
            yield Utils.folder_and_ignore_function_for_sequence(volume.filepath, volume.is_sequence)

        # Textures: images sequence, MP4 or UDIM
        for image in bpy.data.images:
            try:
                # FILE = single image, GENERATED = on Blender?
                if image.source == ("FILE", "GENERATED") or not image.filepath:
                    continue
                if image.library:  # to remove once library will be supported
                    log.debug("Library texture: " + image.filepath)
                if image.source == "SEQUENCE":
                    log.debug(" Sequence: " + image.filepath)
                    yield Utils.folder_and_ignore_function_for_sequence(image.filepath, True)
                    continue

                if image.source == "MOVIE":
                    _, img_ext = os.path.splitext(bpy.path.abspath(image.filepath))
                    if img_ext.upper() == ".MP4":
                        log.debug(" Movie: " + image.filepath)
                        yield Utils.folder_and_ignore_function_for_sequence(image.filepath)
                        continue
                    log.warning("Video texture format no supported: " + image.filepath)
                    continue

                folder, filename = splitpath(image.filepath)
                filename = re.escape(filename)

                if image.source == "TILED":  # UDIM
                    log.debug(" UDIM: " + image.filepath)
                    # https://docs.blender.org/manual/en/3.2/modeling/meshes/uv/workflows/udims.html?highlight=udim#file-substitution-tokens
                    pattern = filename.replace("<UDIM>", "\d+").replace("<UVTILE>", "u\d+_v\d+")
                    if pattern != filename:  # <UDIM> or <UVTILE> found: use the pattern
                        yield folder, Utils.include_files_matching(pattern)
                        continue

                    # fallback to replacing digit sequences
                    yield Utils.folder_and_ignore_function_for_sequence(image.filepath, True)
                    continue
                log.debug(" Plain Texture TEX_IMAGE: " + image.source + " " + image.filepath)

            except AttributeError:
                pass

        for object in bpy.data.objects:
            if object.library or object.type == "VOLUME":
                #     log.warning(" Subscene not yet supported: %s", object.library.filepath)
                continue

            # cache
            for modifier in object.modifiers:
                if modifier.type == "MESH_CACHE":  # MDD ou PC2
                    yield Utils.folder_and_ignore_function_for_sequence(modifier.filepath)
                elif modifier.type == "MESH_SEQUENCE_CACHE":  # certain ABC et USD
                    #     yield Utils.folder_and_ignore_function_for_sequence(
                    #         modifier.cache_file.filepath, modifier.cache_file.is_sequence
                    #     )
                    continue
                else:
                    try:
                        yield modifier.domain_settings.cache_directory, None
                    except AttributeError:
                        pass

            # vdb must be manually handled
            # because bpy.ops.file.pack_all() does not work with vdb sequence
            # (only the first is packed)
            # if object.type == "VOLUME":
            #     yield Utils.folder_and_ignore_function_for_sequence(object.data.filepath, object.data.is_sequence)

            # Textures: images sequence, MP4 or UDIM
            if object.type == "MESH":
                continue
                # for mat_slot in object.material_slots:
                # try:
                # for node in mat_slot.material.node_tree.nodes:
                # if node.type != "TEX_IMAGE" or not node.image.filepath:
                #     continue

                # if node.image.source == "SEQUENCE":
                #     log.debug(" Sequence: " + node.image.filepath)
                #     yield Utils.folder_and_ignore_function_for_sequence(node.image.filepath, True)
                #     continue

                # if node.image.source == "MOVIE":
                #     _, img_ext = os.path.splitext(bpy.path.abspath(node.image.filepath))
                #     if img_ext.upper() == ".MP4":
                #         log.debug(" Movie: " + node.image.filepath)
                #         yield Utils.folder_and_ignore_function_for_sequence(node.image.filepath)
                #         continue
                #     log.warning("Video texture format no supported: " + node.image.filepath)
                #     continue

                # folder, filename = splitpath(node.image.filepath)
                # filename = re.escape(filename)

                # if node.image.source == "TILED":  # UDIM
                #     log.debug(" UDIM: " + node.image.filepath)
                #     # https://docs.blender.org/manual/en/3.2/modeling/meshes/uv/workflows/udims.html?highlight=udim#file-substitution-tokens
                #     pattern = filename.replace("<UDIM>", "\d+").replace("<UVTILE>", "u\d+_v\d+")
                #     if pattern != filename:  # <UDIM> or <UVTILE> found: use the pattern
                #         yield folder, Utils.include_files_matching(pattern)
                #         continue

                #     # fallback to replacing digit sequences
                #     yield Utils.folder_and_ignore_function_for_sequence(node.image.filepath, True)
                #     continue

                # log.debug(" Plain Texture TEX_IMAGE: " + node.image.source + " " + node.image.filepath)

                # except AttributeError:
                #    pass

    @staticmethod
    def copy_additional_folders(folders, destination, log):
        """
        folders = list of (folder, ignore_function)
        """
        for folder, ignore_function in folders:
            if folder == "":
                continue

            if folder.startswith("//") and ".." not in folder:
                # relative path, without "..": keep the same structure
                dst = folder[2:]
            else:
                # absolute path, find part of path without "."
                # append md5 to basename

                dst = md5_basename(folder)

            src = os.path.abspath(bpy.path.abspath(folder))
            if not os.path.exists(src):
                log.warning(" Path not found: %s", src)
                continue

            dst = os.path.join(destination, dst)
            os.makedirs(dst, exist_ok=True)
            try:
                shutil.copytree(src=src, dst=dst, dirs_exist_ok=True, ignore=ignore_function)
            except FileNotFoundError:
                log.warning("Unable to copy '%s': folder not found", src)

    @staticmethod
    def convert_format(file_format):
        static_formats = {
            "AVI_JPEG": "AVI_JPEG",
            "AVI_RAW": "AVI_RAW",
            "BMP": "bmp",
            "CINEON": "cin",
            "DPX": "dpx",
            "FFMPEG": "FFMPEG",
            "HDR": "hdr",
            "IRIS": "rgb",
            "JPEG": "jpg",
            "JPEG2000": "jpg",
            "OPEN_EXR_MULTILAYER": "exr",
            "OPEN_EXR": "exr",
            "PNG": "png",
            "TARGA_RAW": "tga",
            "TARGA": "tga",
            "TIFF": "tif",
            "WEBP": "webp",
        }

        return static_formats.get(file_format, "unknown")

    @staticmethod
    def remove_temp(temp_path):
        try:
            shutil.rmtree(temp_path)
        except Exception as e:
            logging.error(" %s", e)

    @staticmethod
    def remove_ntfs_prefix(path):
        # Matches the \\?\ prefix for handling long path in windows
        if path.startswith("\\\\?\\UNC"):
            path = path.replace("UNC", "\\", 1)
        regex_rule = (r"^\\\\\?\\", "")
        pattern, replacement = regex_rule
        return re.sub(pattern, replacement, path)

    @staticmethod
    def longname(path):
        normalized = os.fspath(path.resolve())
        # Windows : workaround for 256 char path limit
        if platform.system() == "Windows" and not normalized.startswith("\\\\?\\"):
            if normalized.startswith("\\\\"):
                normalized = "\\\\?\\UNC" + normalized
            else:
                normalized = "\\\\?\\" + normalized
        return pathlib.Path(normalized)

    @staticmethod
    def creat_tempdir(desktop, proj_name):
        desktop_ranch = os.path.join(desktop, "_RANCHecker_" + proj_name)
        status_dir = True
        desktop_ranch = Utils.longname(pathlib.Path(desktop_ranch))
        while os.path.exists(desktop_ranch):
            Utils.remove_temp(desktop_ranch)
            time.sleep(0.15)
        os.makedirs(desktop_ranch, exist_ok=True)
        return str(desktop_ranch)

    @staticmethod
    def save_scene_in_tmpdir(desktop_temp, proj_name):
        new_proj = os.path.abspath(os.path.join(desktop_temp, proj_name))  # blender 3.1 needs absolute path
        save_back = bpy.context.preferences.filepaths.save_version

        bpy.context.preferences.filepaths.save_version = 0
        bpy.ops.wm.save_as_mainfile(filepath=new_proj, copy=True, compress=False, relative_remap=False)
        bpy.context.preferences.filepaths.save_version = save_back

    @staticmethod
    def prepare_archive_path(desktop_temp, proj_name) -> str:
        # resolve : need disable workaround Windows path limit
        Utils.remove_ntfs_prefix(desktop_temp)
        p = (
            pathlib.Path(
                os.path.join(
                    desktop_temp,
                    "../",
                    proj_name,
                )
            )
            .with_suffix(".vub")  # replace extension
            .resolve()  # make the path absolute and remove ".."
        )
        if os.path.exists(p):
            p.unlink(missing_ok=True)  # remove existing archive, if any

        p = Utils.longname(p)
        return str(p)

    @classmethod
    def create_archive(cls, desktop_temp, proj_name):
        archive_path = cls.prepare_archive_path(desktop_temp, proj_name)
        print(f"Archive Path : {Utils.remove_ntfs_prefix(archive_path)}")
        # zip : need disable workaround Windows path limit
        arch_path = Utils.remove_ntfs_prefix(archive_path)
        compress_value = int(RANCHeckerPreferences.singleton().vub_compression_level)
        with zipfile.ZipFile(
            arch_path, "w", zipfile.ZIP_DEFLATED, allowZip64=True, compresslevel=compress_value
        ) as zip_path:
            for root, dirs, files in os.walk(desktop_temp):
                for _file in files:
                    # Error log file will be included last in zip
                    # - (prevent python error if written twice) -
                    root = Utils.longname(pathlib.Path(root))
                    if os.path.basename(_file) != "RANCHecker.log":
                        zip_filename = os.path.join(root, _file).replace(desktop_temp, "")
                        zip_path.write(os.path.join(root, _file), arcname=zip_filename)
        return str(arch_path)

    @classmethod  # ----- RENDERMAN ----
    def create_archive_using_renderman_packer(cls, desktop_temp, proj_name, log):
        """
        Creates the .vub using the bpy.ops.renderman.scene_package operator
        """
        file_path = cls.prepare_archive_path(desktop_temp, proj_name)

        # RenderMan packing uses zip : need disable workaround Windows path limit
        file_path = Utils.remove_ntfs_prefix(file_path)

        # Renderman packing in a dedicated directory (must be empty)
        rm_pack_dir = os.path.join(desktop_temp, "rm_pack")
        os.makedirs(rm_pack_dir, exist_ok=True)

        rdm_version = [mod for mod in addon_utils.modules() if mod.bl_info["name"] == "RenderMan For Blender"][
            0
        ].bl_info["version"]

        # bpy.ops.wm.revert_mainfile() in Renderman packer loose some data like current view_layer
        if (
            RANCHeckerPreferences.singleton().use_tex_rman_packer
            and float(str(rdm_version[0]) + "." + str(rdm_version[1])) > 26.2
        ):
            log.debug("RenderMan Packer: Use tex enabled")
            bpy.ops.renderman.scene_package(directory=rm_pack_dir, filepath=file_path, use_tex=True)
        else:
            bpy.ops.renderman.scene_package(directory=rm_pack_dir, filepath=file_path)
        # exemple:
        #  bpy.ops.renderman.scene_package(
        #   directory="C:\\Users\\Ranch\\Desktop\\simpleTest\\RM_pack\\",
        #   filepath="C:\\Users\\Ranch\\Desktop\\simpleTest\\RM_pack\\TestBasicOpt3_udim.zip")

        return file_path

    @staticmethod  # ----- BAT ----
    def create_archive_using_blender_asset_tracer(desktop_temp: str, proj_name: str, logger: logging.Logger):
        """
        Copy all project files in desktop_temp using the Blender Asset Tracer
        https://projects.blender.org/blender/blender-asset-tracer
        """
        # pylint: disable-next=import-outside-toplevel,import-error
        from .wheels import load_module

        logger.debug("BAT: load_module")
        _, bat_pack, bat_blendfile = load_module("blender_asset_tracer", ("pack", "blendfile"))
        blend_file = pathlib.Path(bpy.data.filepath).resolve()
        project_path = blend_file.parent

        bat_logger = logging.getLogger("blender_asset_tracer")
        bat_logger.setLevel(logging.DEBUG)

        class InfoToDebugHandler(logging.StreamHandler):
            """
            blender_asset_tracer INFO logs are quite verbose.
            Change the level to DEBUG before forwarding them to the upstream logger.
            """

            def __init__(self, upstream: logging.Logger, keep_info_level: bool = False):
                super().__init__()
                self.keep_info_level = keep_info_level
                self.upstream = upstream

            def emit(self, record: logging.LogRecord):
                if not self.keep_info_level and record.levelno == logging.INFO:
                    record.levelno = logging.DEBUG
                self.upstream.handle(record)

        # reset BAT logger handlers
        current_handler = InfoToDebugHandler(
            logger,
            RANCHeckerPreferences.singleton().verbose_logging,
        )
        previous_handlers = bat_logger.handlers
        for hdlr in previous_handlers:
            bat_logger.removeHandler(hdlr)
        bat_logger.addHandler(current_handler)
        try:
            with bat_pack.Packer(
                blend_file,
                project_path,
                desktop_temp,
                noop=False,
                compress=False,
                relative_only=False,
            ) as packer:
                logger.debug("BAT: strategise")
                packer.strategise()

                # hack in case of special chars in blend name
                # adjust the BAT actions to rename the blend_file
                if proj_name != blend_file.name:
                    logger.debug("BAT: renaming blend")
                    # pylint: disable-next=protected-access
                    packer._actions[blend_file].new_path = packer._actions[blend_file].new_path.with_name(
                        proj_name
                    )

                logger.debug("BAT: execute")
                packer.execute()

            return blend_file
        finally:
            bat_logger.removeHandler(current_handler)
            for hdlr in previous_handlers:
                bat_logger.addHandler(hdlr)
            bat_blendfile.close_all_cached()

    @staticmethod
    def get_renderer(scene):
        renderer = scene.render.engine
        device = "CPU"
        if renderer == "CYCLES" and scene.cycles.device == "GPU":
            device = "GPU"
        elif renderer in ("BLENDER_EEVEE", "BLENDER_EEVEE_NEXT", "octane", "REDSHIFT"):
            device = "GPU"
        return renderer, device

    @staticmethod
    def get_output_format(scene):
        out_format = Utils.convert_format(scene.render.image_settings.file_format)

        if scene.render.engine == "PRMAN_RENDER":
            out_format = Utils.prman_beauty_extension(scene)

            if out_format not in Utils.RENDERMAN_OUTPUT_FORMATS:
                out_format = "exr"

        elif out_format == "unknown":
            out_format = "png"
        return out_format

    @staticmethod
    def get_output_path(scene):
        if scene.render.filepath.strip(" ") == "":
            return ""
        else:  #  [!] os.path.abspath("") renvoie le path de l'executable
            return os.path.abspath(bpy.path.abspath(scene.render.filepath.strip(" ")))

    @staticmethod
    def check_name(out_name, log: logging.Logger, log_out=True):
        if log and Utils.OUTPUT_NAME_FORBIDDEN_CHARS.search(out_name) and log_out:
            log.warning("Special character in outname not supported, will be replaced with '_': %s", out_name)
        file_name = Utils.OUTPUT_NAME_FORBIDDEN_CHARS.sub("_", out_name)
        return file_name

    @staticmethod
    def get_output_name(scene, log):
        file_format = Utils.get_output_format(scene)
        out_name = ""
        list_outputs = []

        if scene.render.engine == "PRMAN_RENDER":  # use renderman names workspace
            out_name = Utils.prman_beauty_filepath(scene).replace("#", "")  # add current_viewlayer first
            list_outputs.append(Utils.check_name(os.path.basename(out_name), log))
            for active_viewlayer in Utils.get_active_viewlayers(scene):
                if not active_viewlayer == bpy.context.view_layer:
                    out_name = Utils.prman_beauty_filepath(scene, viewlayer=active_viewlayer).replace("#", "")
                    print(f"out_name={out_name}")
                    checked_outname = Utils.check_name(os.path.basename(out_name), log)
                    if checked_outname not in list_outputs:
                        list_outputs.append(checked_outname)

            # file format is handled by RenderMan
            return list_outputs

        # Other cases (no renderman )
        out_name, _ = os.path.splitext(os.path.basename(bpy.path.abspath(scene.render.filepath)))
        out_name = out_name.strip(" ")

        # No output file name given
        if not out_name or os.path.isdir(scene.render.filepath):
            if bpy.data.filepath:  # .blend file name
                out_name, _ = os.path.splitext(os.path.basename(bpy.data.filepath))
            else:  # new .blend
                out_name = Utils.get_scene_name(scene) + "_output"  # create default name
        list_outputs.append(Utils.check_name(out_name, log) + "_." + file_format.lower())
        return list_outputs

    @staticmethod
    def get_compositing_outputs(scene):
        if not scene.node_tree or not scene.render.use_compositing:  # scene.use_nodes
            return None
        compo_outputs = []
        # links are always connected, no need to check input[x].is_linked
        output_nodetree = [link for link in scene.node_tree.links if link.to_node.type == "OUTPUT_FILE"]
        if not output_nodetree:
            return None
        output_nodes = [
            node
            for node in scene.node_tree.nodes
            if node.type == "OUTPUT_FILE" and node.name in [link.to_node.name for link in output_nodetree]
        ]
        for link in output_nodetree:
            in_node = link.from_node
            # check if the input node is a render layer node
            if not in_node.type == "R_LAYERS":
                continue
            if not scene.view_layers[in_node.layer].use:
                continue
            out_node = link.to_node
            idx = out_node.inputs.values().index(link.to_socket)  # input index
            slot = out_node.file_slots[idx]
            # exr multilayer : node.base_path is the filename
            if out_node.format.file_format == "OPEN_EXR_MULTILAYER":
                slot_name = os.path.basename(out_node.base_path)
                out_format = Utils.convert_format(out_node.format.file_format)
            else:  # other formats :  base_path is the dirname, slot.path the filename
                slot_name = slot.path
                if slot.use_node_format:
                    out_format = Utils.convert_format(slot.format.file_format)
                else:
                    out_format = Utils.convert_format(out_node.format.file_format)
            # RanchSet: output_file node index + "-" + node.base_path + "_" + slot.path + "_"
            slot_name = slot_name.replace(" ", "_")
            if not slot_name.endswith("_"):
                slot_name = slot_name + "_"
            if len(output_nodes) > 1:
                out_node_idx = output_nodes.index(out_node) + 1
                compo_outputs.append(
                    str(out_node_idx)
                    + "-"
                    + os.path.basename(out_node.base_path)
                    + "_"
                    + slot_name
                    + "."
                    + out_format
                )
            else:
                # print(out_node.name, ":: ", slot_name, "_.", out_format)
                compo_outputs.append(slot_name + "." + out_format)
        return compo_outputs

    @staticmethod
    def prman_additionnal_outputs(viewlayer, log):
        string_utils = rfb_mod.rfb_utils.string_utils

        additionnal_outputs = []
        world_tree = bpy.data.worlds["World"].node_tree
        if (
            "RenderMan Sample Filters" in world_tree.nodes
            and not len(world_tree.nodes["RenderMan Sample Filters"].inputs) == 0
        ):
            cryptomattes = [node for node in world_tree.nodes if node.bl_label == "PxrCryptomatte"]
            for crypto in cryptomattes:
                if crypto.is_active and crypto.outputs[0].is_linked:
                    raw_crypto = os.path.basename(world_tree.nodes[crypto.name].filename)
                    if "<layer>" in raw_crypto:
                        raw_crypto = raw_crypto.replace("<layer>", viewlayer)  # warning spaces
                    expand_crypto = string_utils.expand_string(raw_crypto, frame="")
                    out_crypto = Utils.check_name(expand_crypto, log, log_out=False)
                    if out_crypto not in additionnal_outputs:
                        additionnal_outputs.append(out_crypto)
                    else:
                        warn_name = "cryptomatte"
                        if warn_name not in warning_code:
                            log.warning("Multiple Cryptomatte with same name detected")
                            warning_code.append(warn_name)
        return additionnal_outputs

    @staticmethod
    def prman_beauty_filepath(scene, viewlayer=None):
        display_utils = rfb_mod.rfb_utils.display_utils

        # beauty_token = display_utils.get_beauty_filepath(scene)
        # print(beauty_token["filePath"])
        # list_rmandisp_filenames = list()
        main_view = bpy.context.view_layer  # save current window view_layer
        if viewlayer is None:
            viewlayer = main_view
        bpy.context.window.view_layer = viewlayer  # switch to specified view_layer

        rman_display_filename = display_utils.get_beauty_filepath(
            scene,
            use_blender_frame=True,
            expand_tokens=True,
        )["filePath"]

        # all_output
        # if "<layer>" in beauty_token["filePath"]:
        #     main_view = bpy.context.view_layer
        #     active_viewlayers = Utils.get_active_viewlayers(scene)
        #     #     current_viewlayer = bpy.context.view_layer.name  # will be change by packer
        #     for view_l in active_viewlayers:
        #         bpy.context.window.view_layer = view_l
        #         list_rmandisp_filenames.append(
        #             display_utils.get_beauty_filepath(
        #                 scene,
        #                 use_blender_frame=True,
        #                 expand_tokens=True,
        #             )["filePath"]
        #         )

        out_format = Utils.convert_format(scene.render.image_settings.file_format)  # currentlayer only
        bpy.context.window.view_layer = main_view  # revert to main view_layer before any return

        if display_utils.using_rman_displays():
            return rman_display_filename  # single output
            # return list_rmandisp_filenames #list output

        # out_format = Utils.convert_format(scene.render.image_settings.file_format)  # currentlayer only
        # bpy.context.window.view_layer = main_view #revert to main view_layer
        # list_blenderdisp_filename = [
        #     rmandisp_filename.replace("exr", out_format) for rmandisp_filename in list_rmandisp_filenames
        # ]
        return rman_display_filename.replace("exr", out_format)  # single output
        # return list_blenderdisp_filenames #list output

    @staticmethod
    def prman_beauty_extension(scene):
        (_, ext) = os.path.splitext(Utils.prman_beauty_filepath(scene))
        return ext.lstrip(".")

    @staticmethod
    def rs_output_name(scene, viewlayer, log):
        prefix = []
        version = [
            addon.bl_info.get("version", (-1, -1, -1))
            for addon in addon_utils.modules()
            if addon.bl_info["name"] == "Redshift Render Engine"
        ][0]
        fversion = float(str(version[0]) + "." + str(version[1]) + str(version[2]))
        if fversion >= 2025.6:  # new output options
            if scene.rsAOVSettings.addBlendName:
                # deactivate at render time to avoid double scene name
                log.debug("scene.rsAOVSettings.addBlendName = ", scene.rsAOVSettings.addBlendName)
            if scene.rsAOVSettings.addSceneName:
                prefix.append(scene.name)
            if scene.rsAOVSettings.addLayerName:
                prefix.append(Utils.check_name(viewlayer, log, log_out=False))
        output = "_".join((*prefix, "Main_"))
        if scene.rsAOVSettings.userPrefix != "":
            output = output + scene.rsAOVSettings.userPrefix + "_"
        # if scene.rsAOVSettings.AOVEnabled and not scene.rsAOVSettings.asLayeredEXR:
        #     print("list AOVs")
        return output

    @staticmethod
    def get_render_camera(scene):
        return scene.camera

    @staticmethod
    def get_resolution(scene):
        render_scale = Utils.get_render_scale(scene)
        res_x = int((scene.render.resolution_x * render_scale) / 100)
        res_y = int((scene.render.resolution_y * render_scale) / 100)
        return res_x, res_y

    @staticmethod
    def get_strtend_frame(scene):
        return scene.frame_start, scene.frame_end

    @staticmethod
    def get_scene_name(scene):
        return scene.name_full

    @staticmethod
    def get_frame_step(scene):
        frame_step = scene.frame_step
        return frame_step

    @staticmethod
    def get_render_scale(scene) -> int:
        render_scale = scene.render.resolution_percentage
        return render_scale

    @staticmethod
    def get_destination(scene):
        scn_setting = scene.ranch_settings
        proj_name = os.path.basename(bpy.data.filepath)
        name_only, ext_only = os.path.splitext(proj_name)

        if scn_setting.destination_bool:
            # BUG if relative utiliser ->  os.path.abspath(bpy.path.abspath(folder))
            # des_temp = scn_setting.str_destination
            des_temp = os.path.abspath(bpy.path.abspath(scn_setting.str_destination))
            desktop_temp = Utils.creat_tempdir(des_temp, name_only)
            return desktop_temp
        else:
            desktop = get_desktop()
            desktop_temp = Utils.creat_tempdir(desktop, name_only)
            return desktop_temp

    @staticmethod
    def get_test_frames(scene):
        scn_setting = scene.ranch_settings
        start_frame, end_frame = Utils.get_strtend_frame(scene)
        if not end_frame == 0 and end_frame - start_frame > 1 and scn_setting.test_frames_bool:
            middle_frame = int(start_frame + ((end_frame - start_frame) / 2))
            print("using specify frames on 3 frames")
            frames_to_render = f"{start_frame},{middle_frame},{end_frame}"
            print("TEST FRAMES: ", frames_to_render)
            return {
                "type": "specific",
                "specific_frames": frames_to_render,
            }
        print("full frame range")
        return {}

    @staticmethod
    def get_skip_email(scene):
        scn_setting = scene.ranch_settings
        skip_email = scn_setting.skip_email_bool
        print(f"SKIP EMAIL: {'true' if skip_email else 'false'}")
        return skip_email

    @staticmethod
    def get_list_scenes(scene):
        scenes = None
        scn_set = scene.ranch_settings
        if scn_set.multiscene_bool:
            scenes = bpy.data.scenes
        else:
            scenes = scene
        return scenes

    @staticmethod
    def get_active_viewlayers(scene):
        viewlayers = [viewlayer for viewlayer in scene.view_layers if viewlayer.use]
        return viewlayers

    @classmethod
    def get_ranchecker_version(cls):
        return cls.__formatted_version(bl_info["version"])

    @classmethod
    def get_blender_version(cls):
        return cls.__formatted_version(bpy.app.version)

    @staticmethod
    def get_addons_info():
        default_addons = {
            "io_anim_bvh",
            "io_curve_svg",
            "io_mesh_ply",
            "io_mesh_stl",
            "io_mesh_uv_layout",
            "io_scene_fbx",
            "io_scene_gltf2",
            "io_scene_obj",
            "io_scene_x3d",
            "cycles",
            "pose_library",
        }

        enablist = [addon.module for addon in bpy.context.preferences.addons]
        # print (f"Enabled : {enablist} \n")
        addon_dict = {}

        for addon in addon_utils.modules():
            if addon.__name__ in enablist and addon.__name__ not in default_addons:
                mod = addon.__name__
                name = addon.bl_info["name"]
                ver = (
                    ".".join([str(x) for x in addon.bl_info.get("version")]) if "version" in addon.bl_info else ""
                )
                file = addon.__file__
                addon_dict[name] = (ver, mod, file)

        return OrderedDict(sorted(addon_dict.items(), key=lambda t: t[0]))

    @staticmethod
    def __formatted_version(version_iter):
        return ".".join(str(n) for n in version_iter)

    @staticmethod
    def get_multipass(scene, setpass):
        state = False
        try:
            nodes_name = []
            nodes = scene.node_tree.nodes
            for n in nodes:
                nodes_name.append(n.name)
            for node in nodes_name:
                if scene.node_tree.nodes[node].bl_idname == "CompositorNodeOutputFile":
                    if setpass:
                        bpy.context.scene.node_tree.nodes[node].base_path = "C:\\Blender\\Output\\"
                    state = True
        except:
            pass
        return state

    @staticmethod
    def set_log(string):
        if platform.system() == "Darwin":
            return string + "\r"
        elif platform.system() == "Windows":
            return string

    @staticmethod
    def unicode_filename(file_name):
        (root, _) = os.path.splitext(file_name)
        return not Utils.forbidden_chars.search(root)


class RANCH_OT_message(bpy.types.Operator):
    """
    Draw one message line in UI dialog
    """

    bl_idname = "ranch.message"
    bl_label = "Ranch message:"

    def execute(self, context):
        try:
            bpy.ops.wm.save_mainfile(filepath=bpy.data.filepath)
            return {"FINISHED"}
        except:
            return {"FINISHED"}

    def invoke(self, context, event):
        wm = context.window_manager
        return wm.invoke_props_dialog(self, width=450)

    def draw(self, context):
        ranch_setting = bpy.context.scene.ranch_settings
        layout = self.layout
        row = layout.row()
        row.label(text=ranch_setting.message, icon="INFO")


# taken from RANCHSync repo
def send_to_ranchsync(data: object) -> object:
    def recvall(sock):
        BUFF_SIZE = 4096  # 4 KiB
        data = b""
        while True:
            part = sock.recv(BUFF_SIZE)
            data += part
            if len(part) < BUFF_SIZE:
                # either 0 or end of data
                break
        return data

    HOST = "127.0.0.1"
    PORT = 7009
    s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    try:
        s.connect((HOST, PORT))
        json_data = json.dumps(data)
        s.sendall(bytes(json_data, encoding="utf-8"))
        b = recvall(s)
        return json.loads(str(b, "utf-8"))
    finally:
        s.close()


class UnsupportedBlenderVersionOrEngine(Exception):
    def __init__(self, version):
        super().__init__(f"{version} is not supported (yet), please contact us.")
        self.version = version


class SubmitJob:
    uploaded_id = None

    def get_sweb_slug(self, scene):
        scene_renderer = scene.render.engine.lower()
        if scene_renderer.startswith("blender_eevee"):
            scene_renderer = "eevee"
        elif scene_renderer == "cycles":
            scene_renderer += "-" + scene.cycles.device.lower()
        elif scene_renderer == "prman_render":
            scene_renderer = "renderman"
        return scene_renderer

    def get_supported_renderers_list(self):
        """
        Parse Blender API to get all the infos for current render engine which version are equal or higher to current blender version
        https://www.ranchcomputing.com/api/ranchtools/ranchecker/blender.json

        Return:
            A list of dictionary for each version available in API with new key
                "blender_apiid": index of Blender version in API list,
                "engine_apiid": index of render engine for API renderer list for associated Blender version,
                "sweb_id": sweb ID for render engine,

        """
        blender_version = str(bpy.app.version[0]) + "." + str(bpy.app.version[1])
        next_version = False
        engine_slug = self.get_sweb_slug(bpy.context.scene)
        # print("Blender version: ", blender_version)
        backward_support = []
        blender_api = RANCH_OT_sweb_status.supported_releases()
        for release in blender_api:
            # print("now: ", str(release["version"]))
            if blender_version == str(release["version"]):
                next_version = True
            if next_version:
                engine_ver = [
                    (
                        {
                            "blender_apiid": blender_api.index(release),
                            "engine_apiid": release["renderers"].index(engine),
                            "sweb_id": engine["id"],
                        }
                    )
                    for engine in release["renderers"]
                    if engine["slug"] == engine_slug
                ]
                if engine_ver:
                    backward_support.append(engine_ver[0])
        return backward_support

    def get_supported_renderers(self):
        """
        Parse Sweb API to get the dictionnary of supported renderer for current Blender Version.
        """
        blender_version = str(bpy.app.version[0]) + "." + str(bpy.app.version[1])
        for release in RANCH_OT_sweb_status.supported_releases():
            if blender_version == str(release["version"]):
                return release["renderers"]
        raise UnsupportedBlenderVersionOrEngine(str("Blender " + blender_version))

    def get_sweb_rendererID_OLD(self, scene_renderer):
        """
        Return current version renderer API ID
        """
        renderers = self.get_supported_renderers()
        for renderer in renderers:
            if scene_renderer == renderer["slug"]:
                print("sweb id: ", renderer["id"])
                return renderer["id"]
        return 0

    @classmethod
    def set_class_variable(cls, value):
        cls.uploaded_id = value

    def get_sweb_rendererID(self, scene_renderer):  # add unused argument for legacy
        """
        List Sweb renderer ID in API
        Args:
            scene_renderer : sweb slug for render engine
        Return
            list(): sweb ID

        """
        swid = []
        if self.uploaded_id:
            print("already uploaded once re-using the same ID: ", self.uploaded_id)
            swid.append(self.uploaded_id)
        else:
            renderers = self.get_supported_renderers_list()
            for version in renderers:
                swid.append(version["sweb_id"])
            print("sweb_id: ", swid)
        return swid

    def __get_sweb_priority_OLD(self):
        """
        send a request to sweb to retrieve priorities for current renderer
        return list or false
        """
        scene_renderer = self.get_sweb_slug(bpy.context.scene)
        if not scene_renderer:
            return False
        renderer_id = self.get_sweb_rendererID(scene_renderer)
        print("render_id = ", renderer_id)

        legacy_farm_filter = ""
        if not renderer_id and scene_renderer.startswith("cycles-"):
            # try again (sweb has not been migrated yet)
            renderer_id = self.get_sweb_rendererID("cycles")
            legacy_farm_filter = scene_renderer[len("cycles-") :]
            if legacy_farm_filter == "cpu":
                legacy_farm_filter = "power"

        if not renderer_id:
            bpy.context.scene.ranch_settings.message = (
                "Renderer {} not supported (yet), please contact us.".format(scene_renderer)
            )
            print("Renderer not supported: " + scene_renderer)
            return False
        data = {
            "method": "sweb_get_json",
            "url": "api/renderer-releases/{}/priorities".format(renderer_id),
        }
        resp = self.send_to_ranchsync(data)
        if not resp:
            return False

        for farm, priorities in resp["farms"].items():
            if legacy_farm_filter and farm != legacy_farm_filter:
                continue
            yield from reversed(priorities)

    def __get_sweb_priority(self):
        """
        send a request to sweb to retrieve priorities for current renderer
        return list or false
        """
        scene_renderer = self.get_sweb_slug(bpy.context.scene)
        if not scene_renderer:
            return False
        renderer_ids = self.get_sweb_rendererID(scene_renderer)
        print("render_ids = ", renderer_ids)
        if not renderer_ids:
            blender_version = str(bpy.app.version[0]) + "." + str(bpy.app.version[1])
            raise UnsupportedBlenderVersionOrEngine(
                str(scene_renderer.capitalize() + " for Blender " + blender_version)
            )
        for renderer_id in renderer_ids:
            legacy_farm_filter = ""
            if not renderer_id and scene_renderer.startswith("cycles-"):
                # try again (sweb has not been migrated yet)
                renderer_id = self.get_sweb_rendererID("cycles")
                legacy_farm_filter = scene_renderer[len("cycles-") :]
                if legacy_farm_filter == "cpu":
                    legacy_farm_filter = "power"

            if not renderer_id:
                bpy.context.scene.ranch_settings.message = (
                    "Renderer {} not supported (yet), please contact us.".format(scene_renderer)
                )
                print("Renderer not supported: " + scene_renderer)
                return False
            data = {
                "method": "sweb_get_json",
                "url": "api/renderer-releases/{}/priorities".format(renderer_id),
            }
            resp = self.send_to_ranchsync(data)
            if not resp:
                return False

            for farm, priorities in resp["farms"].items():
                if legacy_farm_filter and farm != legacy_farm_filter:
                    continue
                return reversed(priorities)  # even deprecated version can fill priority

    def fill_coll_priorities(self):
        """
        Retrieve priorities from RANCH site
        if successfully retreived,  fill priorities scene collection
        return len(collection)
        """

        """
            First clear collection
        """
        if len(bpy.context.scene.coll_priority_items) != 0:
            bpy.context.scene.coll_priority_items.clear()
        bpy.context.scene.ranch_settings.message = ""

        priorities = self.__get_sweb_priority()  # Send a request to sweb to get RANCH priorities

        if priorities:
            for p in priorities:
                item = ItemPriority(bpy.context.scene.coll_priority_items.add())
                item.priority_id = int(p["id"])
                item.priority_str = (
                    p["name"]
                    + " /Price "
                    + str(p["ghz_h_price"])
                    + " /"
                    + str(p["max_allocated_nodes"])
                    + " "
                    + str(p["nodes_unit"])
                )

        else:
            if bpy.context.scene.ranch_settings.message == "":
                bpy.context.scene.ranch_settings.message = "Please run RANCHSync if you want to submit."
            print(" WARNING: Can't prepare submission: internet not connected or RANCHSync not running.")
            bpy.ops.ranch.message("INVOKE_DEFAULT")
            return 1

        return len(bpy.context.scene.coll_priority_items)

    def submit_sync_OLD(self):
        ranch_setting = bpy.context.scene.ranch_settings
        # Ranchsync : need disable workaround Windows path limit
        archive_path = Utils.remove_ntfs_prefix(ranch_setting.archive_path)

        if not archive_path or not os.path.exists(archive_path):
            ranch_setting.message = "RANCHSync unable to submit: Archive not created or errors during the process"
            return 1
        if len(bpy.context.scene.coll_priority_items) == 0:
            ranch_setting.message = "Unable to get priorities: RANCHSync not running or Renderer not suported yet"
            return 1

        test_frames = Utils.get_test_frames(bpy.context.scene)
        skip_email = Utils.get_skip_email(bpy.context.scene)

        scene_renderer = self.get_sweb_slug(bpy.context.scene)
        renderer_id = self.get_sweb_rendererID(scene_renderer)
        if not renderer_id and scene_renderer.startswith("cycles-"):
            # try again with "cycles" (sweb has not been migrated yet)
            renderer_id = self.get_sweb_rendererID("cycles")

        if not renderer_id:
            ranch_setting.message = "Renderer not supported: " + scene_renderer
            return 1

        resp = self.send_to_ranchsync(
            {
                "method": "create_and_upload",
                "sweb": {
                    "renderer_id": str(renderer_id),
                    "priority_id": str(ranch_setting.priority),
                    "disclaimer": True,
                    "_optional_parameters_see": "",
                    "frames": test_frames,
                    "skip_finished_email": skip_email,
                },
                "upload": {
                    "file_path": archive_path,
                },
                "download": {},  # Bug RanchSync when defaut download path is given (issue 67)
            }
        )
        if not resp:
            return 0
        # print("resp data ", resp)
        return 2

    def submit_sync(self):
        """
        Submit VUB with RANCHSync
        """
        ranch_setting = bpy.context.scene.ranch_settings
        # Ranchsync : need disable workaround Windows path limit
        archive_path = Utils.remove_ntfs_prefix(ranch_setting.archive_path)

        if not archive_path or not os.path.exists(archive_path):
            ranch_setting.message = "RANCHSync unable to submit: Archive not created or errors during the process"
            return 1
        if len(bpy.context.scene.coll_priority_items) == 0:
            ranch_setting.message = "Unable to get priorities: RANCHSync not running or Renderer not suported yet"
            return 1

        test_frames = Utils.get_test_frames(bpy.context.scene)
        skip_email = Utils.get_skip_email(bpy.context.scene)

        scene_renderer = self.get_sweb_slug(bpy.context.scene)
        renderer_ids = self.get_sweb_rendererID(scene_renderer)
        if not renderer_ids:
            blender_version = str(bpy.app.version[0]) + "." + str(bpy.app.version[1])
            raise UnsupportedBlenderVersionOrEngine(blender_version)
        for renderer_id in renderer_ids:
            if not renderer_id and scene_renderer.startswith("cycles-"):
                # try again with "cycles" (sweb has not been migrated yet)
                renderer_id = self.get_sweb_rendererID("cycles")

            if not renderer_id:
                ranch_setting.message = "Renderer not supported: " + scene_renderer
                return 1
            resp = self.send_to_ranchsync(
                {
                    "method": "create_and_upload",
                    "sweb": {
                        "renderer_id": str(renderer_id),
                        "priority_id": str(ranch_setting.priority),
                        "disclaimer": True,
                        "_optional_parameters_see": "",
                        "frames": test_frames,
                        "skip_finished_email": skip_email,
                    },
                    "upload": {
                        "file_path": archive_path,
                    },
                    "download": {},  # Bug RanchSync when defaut download path is given (issue 67)
                }
            )
            if not resp:
                print("This version is not supported anymore, checking next version")
                if renderer_ids.index(renderer_id) < len(renderer_ids):
                    continue
                return 0
            # print("resp data ", resp)
            self.set_class_variable(renderer_id)  # to avoid multi click issue
            return 2

    def send_to_ranchsync(self, data):
        try:
            resp = send_to_ranchsync(data)
            if "error" in resp:
                bpy.context.scene.ranch_settings.message = resp["error"]
                return False
            return resp
        except ConnectionRefusedError as a:
            bpy.context.scene.ranch_settings.message = "Please run RANCHsync V4 to start uploading."
            return False


class RANCH_OT_submit(bpy.types.Operator):
    """
    Submit Job to RANCH (via RANCHSync/sweb)
    """

    bl_idname = "ranch.submit"
    bl_label = "Submit"
    bl_description = "Submit your project on the RANCH site"

    def execute(self, context):
        ranch_setting = bpy.context.scene.ranch_settings
        ranch_setting.message = "Unknown error"
        create = SubmitJob()
        job = create.submit_sync()
        if job == 1:
            bpy.ops.ranch.message("INVOKE_DEFAULT")
            return {"FINISHED"}
        elif job == 2:
            self.report({"INFO"}, "Job uploaded ...")

            return {"FINISHED"}
        else:
            bpy.ops.ranch.message("INVOKE_DEFAULT")

        return {"FINISHED"}


# Custom log levels.
# logging.INFO = 20
# logging.DEBUG = 10
#   => 9 free slots between them
LOG_LEVEL_SUCCESS = logging.INFO - 1
LOG_LEVEL_FAILURE = logging.INFO - 2
LOG_LEVEL_WARNING = logging.INFO - 3


class InterfaceLogHandler(logging.StreamHandler):
    """
    Logging handler to display log messages in Blender UI with an icon.
    """

    def __init__(self, verbose: bool):
        super().__init__()
        self.verbose = verbose

        # clear previous logs
        prefs = RANCHeckerPreferences.singleton()
        prefs.log_lines.clear()
        prefs.log_line_active = 0

    def emit(self, record: logging.LogRecord):
        if not self.verbose and record.levelno <= logging.DEBUG:
            return

        try:
            msg = record.getMessage()
        except AttributeError:
            print("EMPTY_LOG_RECORD", record)
            return

        prefs = RANCHeckerPreferences.singleton()
        prefs.log_line_active = len(prefs.log_lines)

        item = prefs.log_lines.add()
        item.msg = msg.replace("\n", "")

        # See https://docs.blender.org/manual/en/latest/addons/development/icon_viewer.html
        if record.levelno == logging.INFO:
            item.icon = "BLANK1"
        elif record.levelno == logging.ERROR:
            item.icon = "CANCEL"
        elif record.levelno == logging.WARNING:
            item.icon = "ERROR"
        elif record.levelno == logging.DEBUG:
            item.icon = "MODIFIER_DATA"
        elif record.levelno == LOG_LEVEL_SUCCESS:
            item.icon = "KEYTYPE_JITTER_VEC"
        elif record.levelno == LOG_LEVEL_FAILURE:
            item.icon = "KEYTYPE_EXTREME_VEC"
        elif record.levelno == LOG_LEVEL_WARNING:
            item.icon = "KEYTYPE_KEYFRAME_VEC"
        else:
            item.icon = "CHECKMARK"
        if record.exc_info:
            (_, e, _) = record.exc_info
            self.log_exception(prefs.log_lines, e)

    def log_exception(self, log_lines, exc: Exception):
        # Add line for each nested exception
        while exc:
            # when a blender operator raises an exception the structure is lost
            # and we get a RuntimeError with the stracktrace as multiline message...
            if isinstance(exc, RuntimeError):
                for i, msg in enumerate(str(exc).splitlines()):
                    item = log_lines.add()
                    item.msg = msg
                    if i == 0 or msg.startswith(" ") or msg.startswith("Location:"):
                        # blank icon for (usually) non interesting lines
                        item.icon = "BLANK1"
                    else:
                        item.icon = "CANCEL"
            else:
                item = log_lines.add()
                item.msg = str(exc)
                item.icon = "CANCEL"
            exc = exc.__context__  # extract nested exception


class Logger(logging.Logger):
    """
    Logger creates a new logger, outputing to a file (FileHandler) and the blender interface (InterfaceLogHandler).
    """

    def __init__(self, filename):
        super().__init__("ranchecker")

        # fileHandler (Ranchecker.log)
        self.file_handler = logging.FileHandler(filename, encoding="UTF-8")
        self.file_handler.setFormatter(logging.Formatter("%(levelname)s:%(message)s"))
        self.file_handler.setLevel(logging.DEBUG)  # log everything in file
        self.addHandler(self.file_handler)

        # InterfaceHandler (to show to the user)
        prefs = RANCHeckerPreferences.singleton()
        self.addHandler(InterfaceLogHandler(prefs.verbose_logging))

        self.debug("log_file: %s", Utils.remove_ntfs_prefix(filename))

    def close_file(self):
        self.file_handler.close()

    # methods below for a context management (with Logger(...) as logger:)
    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close_file()


unsupported = []
warning_code = []


class ConfigChecks:
    """
    ConfigChecks performs checks on the configuration of the scene (see method has_config_error).
    """

    def __init__(self, logger: logging.Logger):
        self.log = logger

        self.scene = bpy.context.scene
        self.__ranchecker_version()
        self.__blender_version()
        self.all_checks = [
            self.__render_camera,
            self.__render_node,
            self.__resolution_rercentage,
            self.__frame_step,
            self.__render_scale,
            self.__render_region,
            self.__stereoscopy,
            self.__multipass,
            self.__video_format,
            self.__denoiser,
            self.__unknown_format,
            self.__renderman_blenddirtoken,
            self.__renderman_badscenename,
            self.__rs_engine,
            self.__debug_information,
            self.__check_active_viewlayer,
            self.__warning_linked_scenes,
            self.__pack_scene,  # last
        ]

    def __ranchecker_version(self):
        version = Utils.get_ranchecker_version()
        self.log.debug("RANCHecker version: %s", version)

    def __blender_version(self):
        version = Utils.get_blender_version()
        self.log.info("Blender version: %s", version)

    def __render_camera(self):
        camera = Utils.get_render_camera(self.scene)
        if camera is None:
            error_name = "missing camera"
            if error_name not in unsupported:
                unsupported.append(error_name)
            raise ValueError("No camera set for rendering, Define a camera in your Scene Properties")

    def __render_node(self):
        if self.scene.render.use_compositing and self.scene.use_nodes:
            nt_out = [n.name for n in self.scene.node_tree.nodes if n.type in ["COMPOSITE", "OUTPUT_FILE"]]
            if not nt_out:
                self.log.warning("No Render Output Node found, compositing will be disabled")
                warn_name = "missing output node"
                if warn_name not in warning_code:
                    warning_code.append(warn_name)

    def __resolution_rercentage(self):
        res = Utils.get_render_scale(self.scene)
        self.log.info("Scale percentage: %d%%", res)

    def __pack_scene(self):
        if not bpy.data.use_autopack:
            self.log.info("FYI - RANCH Rendering: use File->External Data->Automatically Pack Ressources")

    def __frame_step(self):
        frame_step = Utils.get_frame_step(self.scene)
        if frame_step > 1:
            error_name = "frame step"
            if error_name not in unsupported:
                unsupported.append(error_name)
            raise ValueError(
                "Frame step > 1 are not supported, "
                + "please go to our submission web page to submit only specific frames."
            )

    def __render_scale(self):
        render_scale = Utils.get_render_scale(self.scene)
        if render_scale != 100:
            self.log.warning("Percentage scale is %d%%, your render resolution is not at full size", render_scale)

    def __render_region(self):
        region = self.scene.render.use_border
        crop_region = self.scene.render.use_crop_to_border
        if region and crop_region:
            self.log.warning("Crop Render region is active")

    def __stereoscopy(self):
        stereo = self.scene.render.use_multiview
        if stereo:
            self.log.warning("Stereoscopy is enabled")

    def __multipass(self):
        if Utils.get_multipass(self.scene, False):
            self.log.warning("Multipasses output: enabled")

    def __video_format(self):
        videoFormats = ["AVI_JPEG", "AVI_RAW", "FFMPEG"]
        file_format = Utils.get_output_format(self.scene)
        # wrong_format = [format for format in Utils.get_output_format(self.scene) if format in videoFormats]
        # if wrong_format:
        #    print("add to raise value and remove loop")
        if file_format in videoFormats:
            error_name = "forbidden video output"
            if error_name not in unsupported:
                unsupported.append(error_name)
            raise ValueError(
                "[ {} ] videos are not supported on the farm, only images sequence. Please use image format".format(
                    file_format
                )
            )

    def __denoiser(self):
        gpu_denoiser = False
        if self.scene.render.engine == "CYCLES" and self.scene.cycles.device == "CPU":
            if self.scene.cycles.use_adaptive_sampling:
                denoiser = self.scene.cycles.denoiser
                self.log.info("Denoiser: %s", denoiser)
                if denoiser == "OPTIX":
                    gpu_denoiser = True

        elif self.scene.render.engine == "PRMAN_RENDER":
            display_utils = rfb_mod.rfb_utils.display_utils

            if self.scene.renderman.blender_optix_denoiser:
                self.log.debug("Renderman using Optix blender Denoiser")
                gpu_denoiser = True
            if display_utils.any_dspys_denoise(bpy.context.view_layer):
                self.log.debug("Renderman Denoiser setting")
        elif self.scene.render.engine == "REDSHIFT":
            if self.scene.redshift.denoisingEnabled:
                denoiser = self.scene.redshift.DenoiseEngine.replace("RS_DENOISEENGINE_", "")
                self.log.info("Denoiser: %s", denoiser)

        if gpu_denoiser:
            self.log.warning("Optix not supported on CPU farm")

    def __renderman_badscenename(self):
        if self.scene.render.engine == "PRMAN_RENDER":
            if not Utils.unicode_filename(os.path.basename(bpy.data.filepath)):
                error_name = "forbidden characters"
                if error_name not in unsupported:
                    unsupported.append(error_name)
                raise ValueError(
                    f"Bad Scene name : only chars [a-zA-Z0-9-_] allowed, Please rename your scene : \
                    {os.path.basename(bpy.data.filepath)}"
                )

    def __renderman_blenddirtoken(self):
        if self.scene.render.engine == "PRMAN_RENDER":
            filepath_utils = rfb_mod.rfb_utils.filepath_utils

            if not filepath_utils.get_pref("rman_use_blend_dir_token", True):
                error_name = "rman token path"
                if error_name not in unsupported:
                    unsupported.append(error_name)
                raise ValueError("Please check 'Use blend_dir token' in RenderManForBlender addon preferencies")

    def __rs_engine(self):
        if (
            self.scene.render.engine == "REDSHIFT"
            and self.scene.redshift.RenderingEngine == "RS_RENDERINGENGINE_RT"
        ):
            error_name = "redshift RT"
            if error_name not in unsupported:
                unsupported.append(error_name)
            raise ValueError("Redshift RT is not supported")

    def __debug_information(self):
        self.log.debug("Current Viewlayer: %s", bpy.context.view_layer.name)
        if self.scene.render.engine == "CYCLES":
            self.log.debug("Cycles max Samples: %s", self.scene.cycles.samples)
            if bpy.context.scene.cycles.use_adaptive_sampling:
                self.log.debug("Adaptative Noise %s", round(self.scene.cycles.adaptive_threshold, 5))
            self.log.debug("Cycles Motion Blur: %s", self.scene.render.use_motion_blur)
            if not self.scene.cycles.use_auto_tile:
                self.log.warning(
                    "Use Tiling is disabled, the render could crash on final resolution due to unsufficient memory"
                )
            else:
                self.log.debug("Cycles Tile Size: %s", self.scene.cycles.tile_size)
        elif self.scene.render.engine.startswith("BLENDER_EEVEE"):
            self.log.debug("Eevee Samples: %s", round(self.scene.eevee.taa_render_samples, 5))
            if self.scene.render.engine == "BLENDER_EEVEE_NEXT":
                self.log.debug("Eevee Motion Blur: %s", self.scene.render.use_motion_blur)
            else:
                self.log.debug("Eevee Motion Blur: %s", self.scene.eevee.use_motion_blur)
        elif self.scene.render.engine == "PRMAN_RENDER":
            self.log.debug("RMAN max Samples: %s", self.scene.renderman.hider_maxSamples)
            self.log.debug("RMAN pixel Variance: %s", round(self.scene.renderman.ri_pixelVariance, 5))
            self.log.debug("RMAN Motion Blur: %s", self.scene.renderman.motion_blur)
            self.log.debug("RMAN engine: %s", self.scene.renderman.renderVariant)  # prman = RIS
        elif self.scene.render.engine == "REDSHIFT":
            self.log.debug("RS Bucket quality: %s", self.scene.redshift.BucketQuality)
            self.log.debug("RS threshold: %s", round(self.scene.redshift.UnifiedAdaptiveErrorThreshold, 6))
            self.log.debug("RS Progressive passes: %s", self.scene.redshift.ProgressiveRenderingNumPasses)
            self.log.debug("RS Motion Blur: %s", self.scene.redshift.MotionBlurEnabled)

    def __unknown_format(self):
        scene_outformat = self.scene.render.image_settings.file_format
        out_format = Utils.convert_format(scene_outformat)

        if self.scene.render.engine == "PRMAN_RENDER":
            rdm_display = bpy.context.view_layer.renderman.use_renderman
            print("rdm display: ", rdm_display)

            if rdm_display:  # Flag use RM display system
                self.log.debug("Using Renderman Display")
                out_format = Utils.prman_beauty_extension(self.scene)
                if out_format not in Utils.RENDERMAN_OUTPUT_FORMATS:
                    self.log.warning(
                        "[ %s ] not yet supported, forced to exr single layer. \
                        Please contact us contact@ranchcomputing.com",
                        scene_outformat,
                    )
                    return
                if (  # EXR : Allow OPEN_EXR_MULTILAYER only if done in compositing
                    out_format == "exr" and scene_outformat == "OPEN_EXR_MULTILAYER"
                ):
                    if self.scene.render.use_compositing and self.scene.node_tree:
                        for node in self.scene.node_tree.nodes:
                            if node.type == "OUTPUT_FILE" and node.format.file_format == "OPEN_EXR_MULTILAYER":
                                return

                    self.log.warning(
                        "[ %s ] not yet supported, forced to exr single layer. \
                        Please contact us contact@ranchcomputing.com",
                        scene_outformat,
                    )
            else:
                self.log.debug("Using Blender Display")

        elif out_format == "unknown":
            self.log.warning(
                "[ %s ] not yet supported, forced to png. Please contact us contact@ranchcomputing.com",
                scene_outformat,
            )

    def __check_active_viewlayer(self):
        if not bpy.context.view_layer.use:
            self.log.warning("current viewlayer [%s] is disabled", bpy.context.view_layer.name)
        for view_l in bpy.context.scene.view_layers:
            if view_l.use:
                return
        error_name = "viewlayer disabled"
        if error_name not in unsupported:
            unsupported.append(error_name)
        raise ValueError("No view layer active for rendering, please active at least one")

    def __warning_linked_scenes(self):
        if (
            not bpy.data.libraries
            or not bpy.data.images
            or RANCHeckerPreferences.singleton().use_blender_asset_tracer
        ):
            return
        main_version = float(str(bpy.app.version[0]) + "." + str(bpy.app.version[1]))
        if (main_version > 4.1 and any([img.is_editable == False for img in bpy.data.images])) or (
            main_version <= 4.1 and len(bpy.data.images) > 1
        ):
            self.log.warning(
                "Textures from libraries can't be packed, use BAT or pack resources manually in each subscene"
            )

    def has_config_error(self) -> bool:
        has_error = False
        for check in self.all_checks:
            try:
                check()
            except Exception as e:
                self.log.error(str(e))
                has_error = True
        return has_error

    def has_config_warning(self) -> bool:
        has_warning = False
        if warning_code:
            has_warning = True
        print("warning ", has_warning)
        return has_warning


class RendermanRenderer:
    """
    RendermanRenderer Return the version of RendermanForBlender
    and Renderman ProServer versions
    """

    def log_version(self, log: logging.Logger):
        env_config = rfb_mod.rfb_utils.envconfig_utils

        for mod in addon_utils.modules():
            if mod.bl_info.get("name") == "RenderMan For Blender":
                addon_version = ".".join(str(n) for n in mod.bl_info.get("version"))

        server_version = env_config.envconfig().build_info.version()
        log.debug("RenderManForBlender: %s - Render ProServer: %s", addon_version, server_version)


class RedshiftRenderer:
    """
    RedshiftRenderer class to implement:
        - load addon module
        - log  addon version
        - retrieve additionnal folders for Redshift
    """

    def __init__(self):
        self.__module_name = "redshift"

    def __get_module(self, mod_name):
        for mod in addon_utils.modules():
            if mod.__name__ == mod_name:
                return mod
        return None

    def log_version(self, log: logging.Logger):
        rs_mod = self.__get_module(self.__module_name)
        rs_version = ".".join([str(x) for x in rs_mod.bl_info.get("version")])
        log.debug("Redshift For Blender: %s ", rs_version)

    def additional_folders(self, log: logging.Logger):
        yield from Utils.additional_folders(log)
        yield from self.rs_additional_folders()

    def rs_additional_folders(self):
        """
        Add missing Redshift folders from legacy Utils.additional_folders method
        """
        #  RS IES light
        for light in bpy.data.lights:
            try:
                if light.redshift.rs_light_type == "LIGHT_IES":
                    for l_node in light.node_tree.nodes:
                        # IES texture external file
                        if (
                            l_node.__class__.__name__ == "RS_PT_LightIESShaderNode"  # RS IES shader node class
                            and l_node.inputs[1].default_value  #  True if IES light is ON
                        ):
                            # l_node.inputs[2].default_value is ies filepath
                            yield Utils.folder_and_ignore_function_for_sequence(l_node.inputs[2].default_value)
            except AttributeError:
                pass
        #  RS Proxy
        for obj in bpy.data.objects:
            try:
                if obj.redshift.type == "RS_PLACEHOLDER" and obj.rsProxy.setProxyEnabled:
                    yield Utils.folder_and_ignore_function_for_sequence(obj.rsProxy.setProxyFromFile)
            except AttributeError:
                pass


class CompanionFileContent:
    def __init__(self, scene, log: logging.Logger):
        self.scene = scene
        self.log = log

    def outputs_lists(self):
        outputs_name = Utils.get_output_name(self.scene, self.log)
        compo_outputs = Utils.get_compositing_outputs(self.scene)
        print(compo_outputs)
        if self.scene.render.engine.startswith(("CYCLES", "BLENDER_EEVEE")) and compo_outputs != None:
            outputs_name.extend(compo_outputs)
        if self.scene.render.engine == "PRMAN_RENDER":
            for view_l in Utils.get_active_viewlayers(self.scene):
                add_output = Utils.prman_additionnal_outputs(view_l.name, self.log)
                if add_output not in outputs_name:
                    outputs_name.extend(add_output)
        if self.scene.render.engine == "REDSHIFT":
            (base, ext) = outputs_name[0].split(".")
            outputs_name.clear()
            for view_l in Utils.get_active_viewlayers(self.scene):
                outputs_name.append(
                    base + Utils.rs_output_name(self.scene, view_l.name, self.log) + "." + ext
                )  # move from Utils to RedshiftRenderer?
        for output in outputs_name:
            yield output

    def ranchecker_info(self, proj_name: str):
        renderer, device = Utils.get_renderer(self.scene)
        render_engine = renderer + "-" + device
        yield render_engine
        self.log.info("Render engine: %s", render_engine)
        if renderer == "BLENDER_EEVEE":
            self.log.warning(
                "EEVEE is not multi GPU, your price estimation may be inaccurate. Please submit a few frames first."
            )

        outputs_name = Utils.get_output_name(self.scene, None)
        if renderer == "REDSHIFT":
            (base, ext) = outputs_name[0].split(".")
            outputs_name[0] = base.removesuffix("_") + "." + ext
        output_path = "C:\\Blender\\Output\\" + outputs_name[0]
        yield output_path
        self.log.info("Output format: %s", outputs_name[0])

        _x, _y = Utils.get_resolution(self.scene)
        resolution = f"{_x} x {_y}"
        yield resolution
        self.log.info("Resolution: %s", resolution)

        start_frame, end_frame = Utils.get_strtend_frame(self.scene)
        frame_range = f"{start_frame} - {end_frame}"
        yield frame_range
        self.log.info("Frame range: %s", frame_range)

        scene_name = Utils.get_scene_name(self.scene)
        yield scene_name
        self.log.info("Current scene: %s", scene_name)

        if (
            bpy.context.scene.render.engine == "REDSHIFT"
            and bpy.context.scene.rsColorSettings.OCIORenderSpaces == "ACEScg"
        ):  # REDSHIFT : ACES by default - own config.ocio file
            # pylint: disable-next=pointless-string-statement
            r"""
            - Lancer Blender en mode ACES pour Redshift -
            Variable d'environnement à positionner avant de lancer Blender.exe
            Power shell -
                ($env:OCIO = 'C:/ProgramData/Redshift/Plugins/Blender/ocio/config.ocio';
                C:/blender/304/blender.exe ;Remove-Item Env:OCIO
            .bat:
                Set OCIO=%ProgramData%\Redshift\Plugins\Blender\ocio\config.ocio
                C:\Blender\304\blender.exe

            """
            self.log.info("OCIO = %s", os.getenv("OCIO"))  # ne pas forcer ACES
            yield "ACES_REDSHIFT"

        elif self.scene.display_settings.display_device == "ACES":
            if bpy.context.scene.render.engine == "PRMAN_RENDER":
                # pylint: disable-next=pointless-string-statement
                r"""
                - Lancer Blender en mode ACES pour RenderMan -
                Variable d'environnement à positionner avant de lancer Blender.exe

                Power shell -
                    ($env:OCIO = -join($env:RMANTREE, 'lib\ocio\ACES-1.2\config.ocio'));
                    C:/blender\306/blender.exe ;Remove-Item Env:OCIO

                .bat:
                    Set OCIO=%RMANTREE%lib\ocio\ACES-1.2\config.ocio
                    C:\Blender\306\blender.exe:

                """
                self.log.info("OCIO = %s", os.getenv("OCIO"))
                yield "ACES_RENDERMAN"
            else:
                yield "ACES"
            self.log.info("Color Management: ACES")
        else:
            yield ""
            self.log.info(
                "Color Management: %s - %s",
                self.scene.display_settings.display_device,
                self.scene.view_settings.view_transform,
            )

        # output blendfile (allows renaming the vub file / mutliple blend in root folder)
        yield proj_name
        if not proj_name.endswith(".blend"):
            _, ext = os.path.splitext(proj_name)
            self.log.warning(
                "Extension %s of the main scene is suspicious",
                ext,
            )
            self.log.warning("Are you sure you want to render a backup file?")


class RANCH_OT_create_archive(bpy.types.Operator):
    """
    Create archive (.vub) for the RANCH
    """

    bl_idname = "ranch.create_archive"
    bl_label = "Create Archive"
    bl_description = "Create your .vub archive for the RANCH"

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        unsupported.clear()
        warning_code.clear()
        self.file_path = bpy.data.filepath
        filename, ext = os.path.splitext(os.path.basename(bpy.data.filepath))
        self.proj_name = Utils.forbidden_chars.sub("_", filename) + ext
        self.name_only, ext_only = os.path.splitext(self.proj_name)
        if bpy.data.filepath == "":  # empty bl filepath means the file is not saved
            self.file_path = "untitled"
            self.proj_name = "untitled.blend"
            self.name_only = "untitled"

        self.desktop_temp = None
        if bpy.context.scene.render.engine == "PRMAN_RENDER":
            # Manage RenderManForBlender-dev imports
            rdm_module = [
                mod.__name__
                for mod in addon_utils.modules()
                if mod.bl_info["name"] == "RenderMan For Blender" and addon_utils.check(mod.__name__)[0] is True
            ][0]
            # pylint: disable-next=import-outside-toplevel,import-error
            global rfb_mod
            rfb_mod = __import__(rdm_module)

    # pylint: disable-next=too-many-statements
    def execute(self, context):
        self.desktop_temp = Utils.get_destination(bpy.context.scene)
        logger = Logger(os.path.join(self.desktop_temp, "RANCHecker.log"))
        logger.debug("desktop_temp: %s", Utils.remove_ntfs_prefix(self.desktop_temp))

        # run settings checks
        if ConfigChecks(logger).has_config_error():
            self.report({"ERROR"}, "Unsupported settings")
            logger.log(LOG_LEVEL_FAILURE, "Unsupported settings %s , read log", unsupported)
            logger.close_file()
            # Last: remove temporary directory
            Utils.remove_temp(self.desktop_temp)
            return {"FINISHED"}

        # attempt to pack all assets
        self.pack_all(logger)

        # blend_paths debugging infos
        blend_paths = dict.fromkeys(bpy.utils.blend_paths(packed=True))  # Remove duplicates
        logger.debug("%d blend_paths:", len(blend_paths))
        for path in blend_paths:
            logger.debug(path)

        vub_filepath = ""
        bpy.context.scene.ranch_settings.archive_path = ""

        try:
            # save as, scene may be reloaded: actualize handler
            self.save_mainscene(logger)

            if bpy.context.scene.render.engine == "PRMAN_RENDER":
                # Renderman
                renderer = RendermanRenderer()
                renderer.log_version(logger)
                # companions files must be written before due to reload by bpy.ops.wm.revert_mainfile() in Renderman packer
                self.write_companion_files_to_folder(self.desktop_temp, logger)
                vub_filepath = Utils.create_archive_using_renderman_packer(
                    self.desktop_temp, self.proj_name, logger
                )
                logger.debug("Renderman archive created")
                # Write  .info, .txt in vub
                self.zip_companion_files(vub_filepath, self.desktop_temp)

            else:
                # Other renderers
                if (
                    RANCHeckerPreferences.singleton().use_blender_asset_tracer
                ):  # BLENDER ASSET TRACER (CYCLES, EEVEE) PACKER
                    logger.info("BAT: start")
                    Utils.create_archive_using_blender_asset_tracer(self.desktop_temp, self.proj_name, logger)
                    logger.debug("BAT: done")
                    self.write_companion_files_to_folder(self.desktop_temp, logger)
                else:  # LEGACY RANCH ASSET PACKER
                    self.savescene_in_tmpdir()
                    folders = None  # Utils.additional_folders(logger)

                    if bpy.context.scene.render.engine == "REDSHIFT":  # REDSHIFT
                        rs_renderer = RedshiftRenderer()
                        rs_renderer.log_version(logger)
                        folders = rs_renderer.additional_folders(logger)
                    else:
                        folders = Utils.additional_folders(logger)

                    self.write_companion_files_to_folder(self.desktop_temp, logger)
                    Utils.copy_additional_folders(folders, self.desktop_temp, logger)

                vub_filepath = Utils.create_archive(self.desktop_temp, self.proj_name)

        except Exception as e:
            self.report({"ERROR"}, "Failed to create archive")
            logger.log(LOG_LEVEL_FAILURE, "Failed to create archive:", exc_info=e)
            logger.close_file()

            # Output log to stdout for easier CLI debugging
            with open(logger.file_handler.baseFilename, "r", encoding="utf-8") as file:
                shutil.copyfileobj(file, sys.stdout)

            # Last: remove temporary directory
            Utils.remove_temp(self.desktop_temp)
            return {"FINISHED"}

        if ConfigChecks(logger).has_config_warning():
            self.report({"WARNING"}, "complete with Warning")
            logger.log(LOG_LEVEL_WARNING, "VUB Archive created with warning, read log")
        else:
            logger.log(LOG_LEVEL_SUCCESS, "VUB Archive successfully created:")
        # user info  : need disable workaround Windows path limit
        # abspath : get rid off ../ in path
        logger.info(os.path.abspath(Utils.remove_ntfs_prefix(vub_filepath)))

        self.archive_success(logger)

        logger.close_file()

        # Write log at the end in vub
        self.write_log_to_zip(vub_filepath)

        # Last: remove temporary directory
        Utils.remove_temp(self.desktop_temp)

        bpy.context.scene.ranch_settings.archive_path = vub_filepath  # used for RANCHSync
        return {"FINISHED"}

    def archive_success(self, logger: logging.Logger):
        """
        Archive created successfully: prepare submit by getting the priorities from sweb.
        """
        sj = SubmitJob()
        try:
            sj.fill_coll_priorities()
        except Exception as e:
            logger.warning("Unable to get priorities on RANCH", exc_info=e)

        msg = bpy.context.scene.ranch_settings.message
        if msg:
            logger.warning(msg)
            self.report({"WARNING"}, msg)

            # - to solve -
            # Handlers are cleared when bpy.ops.wm.revert_mainfile()/save/saveas is called
            # (INVOKE_DEFAULT doesn't work)-> EXEC_DEFAULT
            # https://blender.stackexchange.com/questions/6101/
            # poll-failed-context-incorrect-example-bpy-ops-view3d-background-image-add/6105#6105
            bpy.ops.ranch.message("EXEC_DEFAULT")

    @staticmethod
    def pack_all(log):
        log.debug("bpy.ops.file.pack_libraries")
        try:
            bpy.ops.file.pack_libraries()
        except RuntimeError as e:
            for msg in str(e).splitlines():
                log.warning(msg)
        log.debug("bpy.ops.file.pack_all")
        try:
            # BAT : prefered behavior -> pack external references outside .blend
            if not RANCHeckerPreferences.singleton().use_blender_asset_tracer:
                bpy.ops.file.pack_all()
                # if len(bpy.data.libraries) >= 1:
                #     for img in bpy.data.images:
                #         if img.library:
                #             # img.override_create() #do not connect override with objects in the scenes
                #             img.override_hierarchy_create(
                #                 bpy.context.scene, bpy.context.view_layer, do_fully_editable=True
                #             )  # created but with new instance

        except RuntimeError as e:
            for msg in str(e).splitlines():
                log.warning(msg)

    def save_mainscene(self, logger):
        if bpy.data.is_dirty or not bpy.data.is_saved:
            # Changes have been made since the last save. Display message.
            # ,bpy.context.scene.ranch_settings.message = "The scene has been automatically saved."
            logger.info("The scene has been automatically saved.")
            # bpy.ops.ranch.message("INVOKE_DEFAULT")
        bpy.ops.wm.save_mainfile(filepath=bpy.data.filepath)

    def savescene_in_tmpdir(self):
        Utils.save_scene_in_tmpdir(self.desktop_temp, self.proj_name)

    def companion_files_contents(self, log):
        def join_lines(lines):
            return "\r\n".join(lines) + "\r\n"

        companionFileContent = CompanionFileContent(bpy.context.scene, log)
        yield ("Addons_info.json", str(json.dumps(Utils.get_addons_info(), indent=4)))
        yield ("Outputs_list.txt", join_lines(companionFileContent.outputs_lists()))
        yield ("RANCHecker.info", join_lines(companionFileContent.ranchecker_info(self.proj_name)))

    def write_companion_files_to_folder(self, folder, log):
        """
        Write companions files to a specific folder using logger
        args:
        folder : destination of the companions files
        log : information to write
        """
        for filename, content in self.companion_files_contents(log):
            with open(os.path.join(folder, filename), "w", newline="") as f:
                f.write(content)

    # def write_companion_files_to_zip(self, archive_filepath, log):
    #     with zipfile.ZipFile(  # zip : need disable workaround Windows path limit
    #         archive_filepath.replace("\\\\?\\", ""), "a", zipfile.ZIP_DEFLATED, allowZip64=True
    #     ) as zip_path:
    #         for filename, content in self.companion_files_contents(log):
    #             zip_path.writestr(filename, content)

    def zip_companion_files(self, archive_filepath, temp_companion_path):
        """
        Zip companion files from a specific folder to an already existing ZIP
        args:
        archive_filepath : full path of the ZIP
        temp_companion_path : directory path of the companions files
        """
        compress_value = int(RANCHeckerPreferences.singleton().vub_compression_level)
        with zipfile.ZipFile(  # zip : need disable workaround Windows path limit
            archive_filepath.replace("\\\\?\\", ""),
            "a",
            zipfile.ZIP_DEFLATED,
            allowZip64=True,
            compresslevel=compress_value,
        ) as zip_path:
            for file in [
                file
                for file in os.listdir(temp_companion_path)
                if os.path.isfile(os.path.join(temp_companion_path, file)) and file != "RANCHecker.log"
            ]:
                full_path = os.path.join(temp_companion_path, file)
                zip_path.write(full_path, arcname=file)

    def write_log_to_zip(self, archive_filepath):
        try:
            compress_value = int(RANCHeckerPreferences.singleton().vub_compression_level)
            with zipfile.ZipFile(  # zip : need disable workaround Windows path limit
                archive_filepath.replace("\\\\?\\", ""),
                "a",
                zipfile.ZIP_DEFLATED,
                allowZip64=True,
                compresslevel=compress_value,
            ) as zip_path:
                zip_path.write(os.path.join(self.desktop_temp, "RANCHecker.log"), arcname="RANCHecker.log")
        except Exception as e:
            print(str(e))


class MultiSceneProp(bpy.types.PropertyGroup):
    scn_name: StringProperty(name="Name", description="item", default="Untitled")
    scn_state: bpy.props.BoolProperty(name="", default=False)


class RANCH_OT_refresh_scenes(bpy.types.Operator):
    bl_idname = "ranch.refresh_scenes"
    bl_label = "Refresh scenes"
    add = bpy.props.BoolProperty(default=True)

    def execute(self, context):
        scns = []
        scns = bpy.data.scenes
        self.clear(context)
        for snc in scns:
            item = context.scene.ranch_scenes.add()
            item.scn_name = str(snc.name)
        return {"FINISHED"}

    def clear(self, context):
        scn = context.scene.ranch_scenes
        while len(scn):
            scn.remove(0)


class View3DPanel:
    bl_category = "RANCHECKER"
    bl_space_type = "VIEW_3D"
    bl_region_type = "UI"

    @classmethod
    def poll(cls, context):
        # return context.object is not None -> Bug ? (context.object contient les objets actifs )
        return context != None


class RANCH_PT_main(View3DPanel, bpy.types.Panel):
    bl_label = "RANCHecker " + " v" + Utils.get_ranchecker_version()

    def draw(self, context):
        scn_set = context.scene.ranch_settings
        layout = self.layout


class RANCH_PT_create_archive(View3DPanel, bpy.types.Panel):
    bl_parent_id = "RANCH_PT_main"
    bl_label = "Prepare project"

    def draw(self, context):
        scn_set = context.scene.ranch_settings
        layout = self.layout

        row = layout.row()
        row.prop(scn_set, "destination_bool")
        if scn_set.destination_bool:
            row = layout.row()
            row.prop(scn_set, "str_destination")

        row = layout.row()
        row.operator("ranch.create_archive", text="Prepare project", icon="PACKAGE")


class RANCH_PT_submit_job(View3DPanel, bpy.types.Panel):
    bl_parent_id = "RANCH_PT_main"
    bl_label = "Upload project"

    def draw(self, context):
        scn_set = context.scene.ranch_settings
        layout = self.layout

        row = layout.row()

        row.prop(scn_set, "priority")
        row = layout.row()
        row.prop(scn_set, "test_frames_bool")
        row = layout.row()
        row.prop(scn_set, "skip_email_bool")
        row = layout.row()
        row.operator("ranch.submit", text="Upload project", icon="EXPORT")


class RANCH_PT_logs(View3DPanel, bpy.types.Panel):
    bl_label = "Logs"

    def draw(self, context):
        layout = self.layout
        prefs = RANCHeckerPreferences.singleton(context)
        layout.template_list("RANCH_UL_logs_list", "", prefs, "log_lines", prefs, "log_line_active")
        layout.operator("ranch.clear_logs", icon="WORDWRAP_OFF")


class RANCH_UL_logs_list(bpy.types.UIList):
    def draw_item(
        self,
        context,
        layout,
        data,
        item,
        icon,
        active_data,
        active_propname,
        index,
        flt_flag,
    ):
        if self.layout_type in {"DEFAULT", "COMPACT"}:
            layout.prop(
                item,
                "msg",
                text="",
                emboss=False,
                expand=False,
                translate=False,
                icon=item.icon,
            )

    def invoke(self, context, event):
        pass


class RANCH_OT_clear_logs(bpy.types.Operator):
    bl_idname = "ranch.clear_logs"
    bl_label = "Clear logs"
    bl_description = "Clear the log window"

    def execute(self, context):
        prefs = RANCHeckerPreferences.singleton(context)
        prefs.log_lines.clear()
        prefs.log_line_active = 0
        return {"FINISHED"}


class RANCH_PT_check_update(View3DPanel, bpy.types.Panel):
    bl_label = "RANCHecker update"

    def draw(self, context):
        layout = self.layout

        layout.row().operator(RANCH_OT_sweb_status.bl_idname, icon="FILE_REFRESH")

        for i, text in enumerate(RANCH_OT_sweb_status.status_lines):
            icon = "NONE"
            if i == 0 and RANCH_OT_sweb_status.icon:
                icon = RANCH_OT_sweb_status.icon
            row = layout.row()
            if i < len(RANCH_OT_sweb_status.status_lines) - 1:
                row.ui_units_y = 0.75  # smaller line height
            row.label(text=text, icon=icon)

        if RANCH_OT_sweb_status.ranchecker_url:
            layout.row().operator(RANCH_OT_upgrade_self.bl_idname, icon="IMPORT")


class RANCH_OT_sweb_status(bpy.types.Operator):
    """
    Retrieve the latest RANCHecker information online
    """

    bl_idname = "ranch.sweb_status"
    bl_label = "Check for update"
    bl_description = "Retrieve the latest RANCHecker information online"

    status_lines = []
    icon = ""
    _data = {}
    _fetched_timestamp = 0.0
    ranchecker_url = ""  # will be filled with the URL when an upgrade is available

    def execute(self, context):
        self.retrieve_online_info()
        return {"FINISHED"}

    @classmethod
    def retrieve_online_info(cls):
        cls.status_lines = ["Checking for an update..."]
        cls.icon = "FILE_REFRESH"
        cls.ranchecker_url = ""
        print("Retrieving latest RANCHecker information online...")
        try:
            # blender becomes unresponsive during this operator.
            # See https://blender.stackexchange.com/a/71830/118994 for possible solutions
            with urllib.request.urlopen(
                "https://www.ranchcomputing.com/api/ranchtools/ranchecker/blender.json"
            ) as response:
                cls._data = json.loads(response.read())
                cls._fetched_timestamp = time.time()
                cls.update_status_text(cls._data["ranchecker"])
        except Exception as e:
            cls.status_lines = [str(e)]
            cls.icon = "ERROR"

    @classmethod
    def update_status_text(cls, ranchecker):
        current_version = bl_info["version"]
        online_version = tuple(map(int, ranchecker["last_version"].split(".")))
        if current_version >= online_version:
            cls.status_lines = ["You are using the latest version."]
            cls.icon = ""
            return

        changelog_lines = ranchecker["last_changelog"].split("\r\n") if ranchecker["last_changelog"] else []
        cls.status_lines = ["Version " + ranchecker["last_version"] + " is available!"] + changelog_lines
        cls.icon = "INFO"
        cls.ranchecker_url = ranchecker["url"]

    @classmethod
    def _retrieve_online_info_if_stale(cls):
        if cls._fetched_timestamp > 0 and time.time() < cls._fetched_timestamp + 60 * 60:
            # online info was fetch successfully less than 1 hour ago: use the cached version
            return
        cls.retrieve_online_info()

    @classmethod
    def release_filename(cls) -> str:
        return cls._data["ranchecker"]["filename"]

    @classmethod
    def supported_releases(cls):
        cls._retrieve_online_info_if_stale()
        return cls._data["application"]["releases"]


class RANCH_OT_upgrade_self(bpy.types.Operator):
    """
    Download and install the latest RANCHecker version
    """

    bl_idname = "ranch.upgrade_self"
    bl_label = "Upgrade RANCHecker"
    bl_description = "Download and install the latest RANCHecker version"

    def execute(self, context):
        with tempfile.TemporaryDirectory(prefix="ranch-") as tmpdir:
            download_path = os.path.join(tmpdir, RANCH_OT_sweb_status.release_filename())
            urllib.request.urlretrieve(RANCH_OT_sweb_status.ranchecker_url, download_path)
            res = bpy.ops.preferences.addon_install(filepath=download_path)
            if res != {"FINISHED"}:
                RANCH_OT_sweb_status.icon = "ERROR"
                RANCH_OT_sweb_status.status_lines = ["Upgrade could not be installed!"]
                return res
            RANCH_OT_sweb_status.icon = "INFO"
            RANCH_OT_sweb_status.status_lines = ["Please restart blender", "to complete the upgrade."]
            RANCH_OT_sweb_status.ranchecker_url = ""
            return res


classes = (
    LogLine,
    RANCHeckerPreferences,
    RANCH_PT_main,
    RANCH_PT_create_archive,
    RANCH_PT_submit_job,
    RANCH_OT_create_archive,
    RANCH_OT_submit,
    ItemPriority,
    RANCH_OT_message,
    SettingsProp,
    MultiSceneProp,
    RANCH_OT_refresh_scenes,
    RANCH_OT_clear_logs,
    RANCH_PT_logs,
    RANCH_UL_logs_list,
    RANCH_PT_check_update,
    RANCH_OT_sweb_status,
    RANCH_OT_upgrade_self,
)

preview_collections = {}


def register():
    from bpy.utils import register_class

    for cls in classes:
        register_class(cls)

    bpy.types.Scene.ranch_scenes = CollectionProperty(type=MultiSceneProp)
    bpy.types.Scene.ranch_scenes_index = IntProperty(default=0)


def unregister():
    from bpy.utils import unregister_class

    for pcoll in preview_collections.values():
        bpy.utils.previews.remove(pcoll)
    preview_collections.clear()
    for cls in classes:
        unregister_class(cls)
    del bpy.types.Scene.ranch_scenes
    del bpy.types.Scene.ranch_scenes_index


if __name__ == "__main__" and "doctest" in sys.argv:
    # wrap in try/except since blender will exit without error on exception unless sys.exit is called
    try:
        import doctest

        (failure_count, test_count) = doctest.testmod()
        print(test_count, "tests passed.")
        assert not failure_count, f"{failure_count} doctest failures"

    # pylint: disable-next=broad-exception-caught
    except Exception as ex:
        _, _, tb = sys.exc_info()
        import traceback

        print("Error: Python: Traceback (most recent call last):", file=sys.stderr)
        traceback.print_tb(tb)
        print(type(ex).__name__ + ":", ex, file=sys.stderr)
        sys.exit(1)
