# coding=utf-8

import os
import re
import sys
from glob import glob
import time
import datetime
import shutil
import socket
import zipfile
import json
import ssl
import webbrowser
import xml.etree.ElementTree as ET

try:
    from urllib.request import Request, urlopen  # python3+
    from urllib.error import URLError, HTTPError
except ImportError:
    from urllib2 import Request, urlopen, URLError  # python2


import maya.cmds as cmds
import maya.utils as utils
import maya.api.OpenMaya as om
import maya.OpenMayaUI as omui
import maya.mel as mel
import xgenm as xg

try:
    from PySide6.QtCore import *
    from PySide6.QtGui import *
    from PySide6.QtWidgets import *
    from shiboken6 import wrapInstance


except ImportError:
    from PySide2.QtCore import *
    from PySide2.QtGui import *
    from PySide2.QtWidgets import *
    from shiboken2 import wrapInstance


ssl._create_default_https_context = ssl._create_unverified_context

# Please do not modify the following line, it is auto completed by Gitlab-Ci
__version__ = "5.2.3"  # replaced by gitlab-ci.yml
VERSION_PATTERN = r"\d+\.\d+\.\d+"

HOST = "127.0.0.1"
PORT = 7009

RANCH_WINDOW = None
MIN_WIDTH = 700
MIN_HEIGHT = 500

COLOR_WARNING = QColor(254, 197, 57)
COLOR_INFO = QColor(82, 133, 166)
COLOR_ERROR = QColor(220, 69, 85)
COLOR_SUCCESS = QColor(66, 174, 90)

try:
    ARNOLD_VERSION = cmds.pluginInfo("mtoa", q=True, pu=True, version=True)
except:
    ARNOLD_VERSION = "0"

if ARNOLD_VERSION != "0":
    from arnold import *
else:
    print("Mtoa is not installed")

if os.name == "nt":
    home = os.getenv("USERPROFILE")
else:
    home = os.getenv("HOME")
DESKTOP = os.path.join(home, "Desktop")

SCENE_PATH = ""
if cmds.file(q=True, sn=True):
    SCENE_PATH = os.path.normpath(os.path.dirname(cmds.file(q=True, sn=True)))
else:
    SCENE_PATH = DESKTOP

try:
    RANCH_PLUGIN_FOLDER = os.path.dirname(cmds.pluginInfo("RANCHecker", q=True, p=True))
except:
    cmds.confirmDialog(title="RANCHecker", m="cannot load RANCHecker.py")

TEMP_PREFIXE = "__RANCH__"

VERSION_URL = "https://www.ranchcomputing.com/api/ranchtools/ranchecker/maya.json"
ICON_RENDERER_URL = "https://www.ranchcomputing.com/assets/imgs/software/_renderer/"


COMMAND_LINE = "C:\\Maya\\{}\\bin\\mayapy.exe " "C:\\Maya\\RANCHBatch.py "
RANCH_ARCHIVE_FILES = [
    "RANCHecker.info",
    "RANCHecker.log",
    "renderer_detected.txt",
    "command_line.txt",
]

UNSUPPORTED_CHARACTER = re.compile(r"[^a-zA-Z0-9_.-]+")

SOURCEIMAGES_PATH = os.path.normpath("C:\\Maya\\Currentjob\\sourceimages")
CACHE_PATH = os.path.normpath("C:\\Maya\\Currentjob\\cache")

SUPPORTED_RENDERER = []


def maya_useNewAPI():
    pass


def maya_window():
    pointer = omui.MQtUtil.mainWindow()
    return wrapInstance(int(pointer), QWidget)


class Api(QObject):
    data = {}

    def __init__(self):
        self.__get_api_data()

    def check_update(self):
        if self.check_version():
            file_name = self.data["ranchecker"]["filename"]
            archive_path = os.path.join(RANCH_PLUGIN_FOLDER, file_name)
            # Download archive
            try:
                req = Request(self.data["ranchecker"]["url"])
                res = urlopen(req)
            except HTTPError as e:
                print(e)
            except URLError as e:
                print(e)
            with open(archive_path, "wb") as f:
                f.write(res.read())

            # Extract
            with open(archive_path, "rb") as fn:
                ext_zip = zipfile.ZipFile(fn)
                ext_zip.extractall(RANCH_PLUGIN_FOLDER)
                fn.close()
            # Remove archive
            try:
                os.remove(archive_path)
                message = "Please restart maya now"
                cmds.confirmDialog(title="RANCHecker", m=message, button=["Ok"])
            except OSError as e:
                print(e)

    def __get_api_data(self):
        try:
            res = urlopen(VERSION_URL)
            self.data = json.loads(res.read())
        except:
            self.data = {}

    def check_version(self):
        """
        Checks if the current version of the software matches the latest version fetched from the API,
        and if it's in correct format (X.X.X). If the versions do not match or the format is incorrect,
        a dialog is prompted for the user to either update or ignore.

        The function performs the following steps:
        - Checks if the software is in development mode (__version__ == "0.0.0").
        - Checks if the self.data dictionary (representing API response) is empty.
        - Fetches version data from 'ranchecker' in self.data.
        - Validates if the fetched version ('last_version') and local version (__version__) follow the correct format, i.e., '\d+\.\d+\.\d+'.
        - If any of the versions is in incorrect format, a dialog is shown with an option to update.
        - Compares local version and fetched version part by part, and if a more recent version is available, a dialog is shown with an option to update.

        Returns:
            bool: False if the current version is incorrect, in dev mode, or if the user chooses to ignore the update.
                  True if the user chooses to update to the latest version.

        Note: Any exceptions raised during the execution of the function are caught and treated as a version checking failure, thus returning False.

        """

        # return false in dev mode
        if __version__ == "0.0.0":
            return False
        # API response must not be empty
        if not self.data:
            return False
        try:
            fetched_version = self.data.get("ranchecker")
            if not fetched_version:
                return False
            changelog = fetched_version.get("last_changelog")
            if not changelog:
                return False
            distant_version = fetched_version.get("last_version")

            # Check that API version format is correct ("\d+\.\d+\.\d+")
            if re.fullmatch(VERSION_PATTERN, distant_version) is None:
                return False

            # Check that local RChecker version format is correct ("\d+\.\d+\.\d+")
            if re.fullmatch(VERSION_PATTERN, __version__) is None:
                message = "RANCHecker version is not correct.\n\nPlease update or download the latest version from www.ranchcomputing.com."
                result = cmds.confirmDialog(
                    title="RANCHecker - Incorrect version",
                    m=message,
                    button=["Update", "Ignore"],
                )
                if result == "Update":
                    return True
                return False

            # Compare version part by part (API & local)
            outdated_version_flag = False
            for fetched_version_digits, local_version_digits in zip(
                distant_version.split("."), __version__.split(".")
            ):
                if int(fetched_version_digits) > int(local_version_digits):
                    outdated_version_flag = True
                    break

            # Check for update if a more recent version is available
            if outdated_version_flag:
                message = "New RANCHecker version {} is available \n\n Changelog: \n\n {}".format(
                    distant_version, changelog
                )
                result = cmds.confirmDialog(
                    title="RANCHecker",
                    m=message,
                    button=["Update", "Ignore"],
                )
                if result == "Update":
                    return True
        except:
            return False
        return False

    def get_support_renderer(self):
        renderer_list = list()
        self.__get_api_data()
        if self.data:
            for release in self.data["application"]["releases"]:
                if str(release["version"]) == str(Utils.get_maya_version()):
                    for renderer in release["renderers"]:
                        renderer_list.append(
                            {
                                "name": renderer["name"],
                                "version": renderer["version"],
                                "id": renderer["id"],
                            }
                        )
        return renderer_list

    def get_support_plugins(self):
        plugins_list = list()
        if self.data:
            for release in self.data["application"]["releases"]:
                if str(release["version"]) == str(Utils.get_maya_version()):
                    for plugin in release["plugins"]:
                        plugins_list.append({"name": plugin["name"], "version": plugin["version"]})
        return plugins_list

    def get_picto_image_url(self):
        if self.data:
            if self.is_picto_active():
                return self.data["marketing"]["img"]
            return self.data["dashboard"]["img"]

    def get_picto_target_url(self):
        if self.data:
            if self.is_picto_active():
                return self.data["marketing"]["target"]
            return self.data["dashboard"]["target"]

    def is_picto_active(self):
        return self.data["marketing"]["active"]


class Utils:
    @staticmethod
    def get_operating_system():
        if cmds.about(windows=True):
            return "Windows"
        elif cmds.about(macOS=True):
            return "Mac Os"
        elif cmds.about(linux=True):
            return "Linux"
        else:
            return "None"

    @staticmethod
    def get_maya_version():
        return cmds.about(version=True)

    @staticmethod
    def get_workspace():
        return cmds.workspace(query=True, rd=True)

    @staticmethod
    def get_render_output():
        scene_name = os.path.splitext(Utils.get_scene_name())[0]  # when scene is not save return empty
        # scene_name = cmds.file(query=True, namespace=True) #when scene is not save return untitled

        if RENDERER == "redshift":
            format_ext = "." + mel.eval("getImfImageExt()")
            rsimage_prefix = os.path.basename(mel.eval("redshiftGetImageFilePrefix()"))
            # rs_output_basename = mel.eval("redshiftGetImageFormatString("", str(Utils.get_render_camera()))") #resolved name

            rs_output_basename = rsimage_prefix.replace("<scene>", scene_name)
            rs_output_basename = rs_output_basename.replace(
                "<Scene>", scene_name
            )  # if token come from another dcc
            return rs_output_basename + format_ext

        if RENDERER == "renderman":
            displays = cmds.ls(et="rmanDisplay")
            for out in displays:
                conn = cmds.listConnections(out + ".displayType")
                if not conn:
                    continue
                out_format = cmds.listConnections(out + ".displayType")[0]
                if out_format:
                    if out == "rmanDefaultDisplay":
                        if out_format.startswith(("d_deepexr", "d_openexr")):
                            return scene_name + ".exr"
                        elif out_format.startswith("d_png"):
                            return scene_name + ".png"
                        elif out_format.startswith("d_tiff"):
                            return scene_name + ".tiff"
                        elif out_format.startswith("d_targa"):
                            return scene_name + ".targa"
                else:
                    return scene_name  # TODO test with scalc

        if RENDERER == "vray":
            vray_output_name = cmds.getAttr("vraySettings.fileNamePrefix")

            vray_ext_format = cmds.getAttr("vraySettings.imageFormatStr")
            # print("Output Name: {}.{}".format(vray_output_name, vray_ext_format))
            if vray_ext_format.startswith("exr"):  # for exr multichannel and deep
                vray_ext_format = "exr"
            if vray_output_name is not None:
                vray_output_name = vray_output_name.replace("<Scene>", scene_name)
                # print(vray_output_name)
                cmds.setAttr(
                    "defaultRenderGlobals.imageFilePrefix",
                    vray_output_name,
                    type="string",
                )
                # vray_output_name = mel.eval("renderSettings -firstImageName")[0].split("/")[-1].split(".")[0]
                return vray_output_name + "." + vray_ext_format
            else:
                return scene_name + "." + vray_ext_format

        # Todo: Add support for custom job name
        else:  # Ar, Sw
            format_ext = "." + mel.eval("getImfImageExt()")
            image_prefix = cmds.getAttr("defaultRenderGlobals.imageFilePrefix")
            if image_prefix is not None:
                output_basename = os.path.basename(image_prefix).replace("<Scene>", scene_name)
            else:
                output_basename = os.path.basename(cmds.renderSettings(gin="<>")[0]).replace(
                    format_ext, ""
                )  # resolved name

            return output_basename + format_ext

    @staticmethod
    def list_api_renderer():
        renderer_list = []
        for engine in SUPPORTED_RENDERER:
            remove = [" ", "-"]
            format_engine = "".join([c for c in engine if c not in remove]).replace("GPU", "").capitalize()
            if format_engine not in renderer_list:
                renderer_list.append(format_engine)
        return renderer_list

    @staticmethod
    def get_current_renderer():
        return cmds.getAttr("defaultRenderGlobals.currentRenderer")

    @staticmethod
    def get_start_frame():
        return int(cmds.getAttr("defaultRenderGlobals.startFrame"))

    @staticmethod
    def get_end_frame():
        return int(cmds.getAttr("defaultRenderGlobals.endFrame"))

    @staticmethod
    def get_framerange_mode():
        if cmds.getAttr("defaultRenderGlobals.animation") and cmds.getAttr(
            "defaultRenderGlobals.startFrame"
        ) != cmds.getAttr("defaultRenderGlobals.endFrame"):
            # animation
            return True
        # still image
        return False

    @staticmethod
    def get_w_resolution():
        return int(cmds.getAttr("defaultResolution.width"))

    @staticmethod
    def get_h_resolution():
        return int(cmds.getAttr("defaultResolution.height"))

    @staticmethod
    def get_scene_name():
        return cmds.file(query=True, shortName=True, sceneName=True)

    @staticmethod
    def get_render_camera():
        cameras = cmds.listCameras()
        for cam in cameras:
            try:
                renderable = cmds.getAttr(cam + ".renderable")
            except ValueError as e:
                print("err ", e)
            if renderable:
                return cam

    @staticmethod
    def get_plugins_list():
        return cmds.pluginInfo(query=True, ls=True)

    @staticmethod
    def is_plugin_loaded(plugin_name):
        if plugin_name in Utils.get_plugins_list():
            return True
        return False

    @staticmethod
    def get_scene_infos():
        infos = {}
        infos["Main scene"] = Utils.get_scene_name()
        infos["Current renderer"] = Utils.get_current_renderer()
        if Utils.get_framerange_mode():  # cmds.getAttr("defaultRenderGlobals.animation"):
            infos["Frame range"] = str(Utils.get_start_frame()) + "-" + str(Utils.get_end_frame())
        else:
            frame = int(cmds.currentTime(query=True))
            infos["Frame range"] = str(frame) + "-" + str(frame)
        infos["Resolution"] = str(Utils.get_w_resolution()) + " X " + str(Utils.get_h_resolution())
        infos["Output"] = Utils.get_render_output()
        infos["Render cameras"] = Utils.get_render_camera()
        return infos

    @staticmethod
    def get_render_farm():
        logic_struct = {
            "arnold": {-1: ["", ""], 0: ["CPU", "ar"], 1: ["GPU", "ar"]},
            "vray": {-1: ["", ""], 0: ["CPU", "vr"], 1: ["GPU", "vr"]},
            "redshift": {-1: ["GPU", "rs"], 0: ["", ""], 1: ["", ""]},
            "renderman": {-1: ["CPU", "rm"], 0: ["", ""], 1: ["", ""]},
            "mayaSoftware": {-1: ["CPU", "sw"], 0: ["", ""], 1: ["", ""]},
            "default": {-1: ["", "unsupported"], 0: ["", ""], 1: ["", ""]},
        }

        renderer = Utils.get_current_renderer()
        render_device = -1

        if renderer not in logic_struct:
            return logic_struct["default"][render_device]

        if renderer == "arnold":
            try:
                if ARNOLD_VERSION and ARNOLD_VERSION.startswith(("3", "2")):
                    render_device = 0
                else:
                    render_device = cmds.getAttr("defaultArnoldRenderOptions.renderDevice")
            except:
                render_device = -1

        if renderer == "vray":
            if cmds.getAttr("vraySettings.productionEngine") == 0:
                render_device = cmds.getAttr("vraySettings.productionEngine")
            else:
                render_device = 1
            # print("DEVICE: ", render_device)

        return logic_struct[renderer][render_device]

    @staticmethod
    def get_stillimage_mode(qt_ui=None):
        if not qt_ui:
            return "Single"
        if qt_ui.tiling_mode.isChecked():
            current_resolution = Utils.get_w_resolution() * Utils.get_h_resolution()
            current_device = Utils.get_render_farm()[0]
            if current_device == "CPU" and current_resolution >= 2073600:
                return "Multi"
            if current_device == "GPU" and current_resolution >= 8294400:
                return "Multi"
        return "Single"

    @staticmethod
    def arnold_legacy_api():
        """
        return legacy API function dictionary for Mtoa 2 and 3
        """
        return {"load": AiASSLoad, "write": AiASSWrite, "mask": AI_NODE_ALL, "universe": []}

    @staticmethod
    def arnold_current_api():
        """
        return current API function dictionary for Mtoa 4+
        """
        return {"load": AiSceneLoad, "write": AiSceneWrite, "mask": None, "universe": [None]}


class Logger(QObject):
    send_log = Signal(object)

    def __init__(self):
        self.__logs = list()

    def log_error(self, msg):
        self.send_log.emit({"ERROR": "  " + msg})
        self._logs = {"ERROR": msg}

    def log_info(self, msg):
        self.send_log.emit({"INFO": "  " + msg})
        self._logs = {"INFO": msg}

    def log_warning(self, msg):
        self.send_log.emit({"WARNING": "  " + msg})
        self._logs = {"WARNING": msg}

    def log_success(self, msg):
        self.send_log.emit({"SUCCESS": "  " + msg})

    def clean_logs(self):
        self.__logs = list()

    @property
    def _logs(self):
        return self.__logs

    @_logs.setter
    def _logs(self, msg):
        self.__logs.append(msg)


RENDERER = cmds.getAttr("defaultRenderGlobals.currentRenderer")  # init at load time


class Worker(QThread, Logger):
    progress_value_text = Signal(str, int)
    progress_range = Signal(int)
    message_emit = Signal(str)

    def __init__(self, ui=None, job_name="", destination=""):
        QThread.__init__(self)
        Logger.__init__(self)

        self.ui = ui
        self.job_name = job_name
        self.destination = destination
        self.saved = True
        self.__temp_folder = ""
        self.renderable_camera = ""
        self.renderable_scene = ""
        self.render_output = ""
        self.vua_archive_path = ""
        self.all_files = list()
        self.prioritys = list()
        self.zip = None
        self.archive_creation_time = 0

        self.quick_check = Quick_check(worker=self)
        # self.list_references = cmds.ls(references=True)
        self.standin_path_list = list()

    # def _renderman_attrib_list(node_name_attr):
    #     """_renderman_attrib_list add extension of a renderman node (depending of its type) ;
    #     all renderman nodes become from the list generate by the function rfm2.txmanager_maya._get_node_list().

    #     Args:
    #         node_name_attr (str): node name attribute
    #     Returns:
    #         str: concatenation of node name & its extension (if it exist for its type ; otherwise return None)
    #     """

    #     # Dict which store extension for each node type
    #     # {node_type: node_extension}
    #     s_types = {
    #         "PxrBumpRoughness": ".b2r_texture",
    #         "PxrDomeLight": ".lightColorMap",
    #         "PxrMultiTexture": [".filename0",".filename1",".filename2",".filename3",".filename4",".filename5",".filename6",".filename7",".filename8",".filename9"]
    #     }

    #     # Get the renderman node type
    #     node_type = cmds.nodeType(node_name_attr)

    #     # cmds.ls(type=ls_types, modified=True)

    #     if node_type in s_types:
    #         # Get the node extension
    #         node_extension = s_types[node_type]
    #         # Return concatenation
    #         return node_name_attr + node_extension
    #     # Node type is not handled
    #     return None

    def _copy_renderman_tex(self):
        if RENDERER == "renderman":
            # try:
            #     import rfm2
            # except Exception as e:
            #     self.log_warning("could not import rfm2, {}".format(e))

            # try:
            #     # rfm2.txmanager_maya.parse_maya_scene() can't be called on client side (modification of the original file)

            #     # NOTE rfm2.txmanager_maya._get_node_list() must be called directly (outside of any thread)

            #     for i in rfm2.txmanager_maya._get_node_list():
            #         attr = self._renderman_attrib_list(i)
            #         if attr:
            #             p = cmds.getAttr(attr)
            #             if not p.endswith(".tex"):
            #                 path = rfm2.txmanager_maya.get_texture_by_path(
            #                     p, i + ".filename", ipr_mode=False, do_dirty_check=False
            #                 )
            #                 self.all_files.append(os.path.normpath(path))
            # except Exception as e:
            #     pass
            #     # self.log_warning("could not get .tex file path, {}".format(e))
            try:
                rman_multitex = cmds.ls(
                    type=[
                        "PxrMultiTexture",
                        "PxrTexture",
                        "PxrBump",
                        "PxrNormalMap",
                        "PxrProjectionLayer",
                        "PxrLayeredTexture",
                        "PxrEnvGround",
                        "PxrPtexture",
                        "PxrBakeTexture",
                        "PxrBakePointCloud",
                        "PxrBumpRoughness",
                        "PxrDomeLight",
                        "PxrRectLight",
                        "PxrDiskLight",
                        "PxrCylinderLight",
                    ]
                )
                for node in rman_multitex:
                    if node in ("PxrDomeLight", "PxrRectLight", "PxrDiskLight", "PxrCylinderLight"):
                        # print(node)
                        path = rfm2.txmanager_maya.get_texture_by_path(
                            node,
                            node + ".lightColorMap",
                            ipr_mode=False,
                            do_dirty_check=False,
                        )
                        if os.path.normpath(path) not in self.all_files:
                            self.all_files.append(os.path.normpath(path))

                    if "PxrBumpRoughness" in node:
                        path = rfm2.txmanager_maya.get_texture_by_path(
                            node,
                            node + ".b2r_texture",
                            ipr_mode=False,
                            do_dirty_check=False,
                        )
                        # print(path)
                        if os.path.normpath(path) not in self.all_files:
                            self.all_files.append(os.path.normpath(path))

                    if "PxrMultiTexture" in node:
                        # print(node)
                        for i in range(10):
                            path = rfm2.txmanager_maya.get_texture_by_path(
                                node,
                                node + ".filename{}".format(i),
                                ipr_mode=False,
                                do_dirty_check=False,
                            )
                            if os.path.normpath(path) not in self.all_files:
                                self.all_files.append(os.path.normpath(path))

                    if node in (
                        "PxrTexture",
                        "PxrBump",
                        "PxrNormalMap",
                        "PxrProjectionLayer",
                        "PxrLayeredTexture",
                        "PxrEnvGround",
                        "PxrPtexture",
                        "PxrBakeTexture",
                        "PxrBakePointCloud",
                    ):
                        # print(node)
                        path = rfm2.txmanager_maya.get_texture_by_path(
                            node,
                            node + ".filename",
                            ipr_mode=False,
                            do_dirty_check=False,
                        )
                        if os.path.normpath(path) not in self.all_files:
                            self.all_files.append(os.path.normpath(path))

            except Exception as e:
                self.log_warning("could not get .tex file path, {}".format(e))

    def remove_unloaded_ref(self, ref_file):
        """
        remove unloaded reference query by maya file command

        Args:
            ref_file (str): full path

        Return:
            Boolean
        """
        orig_path = ref_file
        if ref_file.endswith("}"):
            orig_path = ref_file.split("{")[0]

        if orig_path.endswith((".ma", ".mb")) and orig_path != cmds.file(query=True, sceneName=True):
            if not cmds.referenceQuery(ref_file, isLoaded=True):
                return False
        return True

    def get_sequence_files(self, filepath, token, token_per_digit=True):
        """
        Check files inside the specified path and list the files corresponding to the sequence's token

        Args:
            filepath (str): full file path
            token (str): token used in the path
            token_per_digit (bool): repeated char = True OR full expression = False, default True

        Return:
            list of normalized paths OR None if the specified folder doesn't exist
        """
        dirname = os.path.dirname(filepath)
        if not os.path.exists(dirname):
            self.log_warning("file sequence folder {} does not exist".format(dirname))
            return None
        if token_per_digit:  # ex: ### for 3 digits
            count_seq = filepath.count(token)
            basename, ext = os.path.basename(filepath).split(token * count_seq)
        else:  # ex: %04d for 4 digits
            basename, ext = os.path.basename(filepath).split(token)

        sequence = [
            os.path.join(dirname, file_from_seq)
            for file_from_seq in os.listdir(dirname)
            if file_from_seq.startswith(basename) and file_from_seq.endswith(ext)
        ]
        return sequence

    def get_aiimage_files(self):
        """
        Arnold AiImage node must be query specifically
        if sequence query all the files with the same digits

        Returns:
            sorted list of aiimages OR None
        """
        aiimages = cmds.ls(typ="aiImage")
        if not aiimages:
            return None
        list_files = list()
        for aiimage in aiimages:
            filepath = cmds.getAttr(aiimage + ".filename")
            if cmds.getAttr(aiimage + ".useFrameExtension") and "#" in filepath:
                sequence = self.get_sequence_files(filepath, "#")
                for filepath in sequence:
                    if filepath not in list_files:
                        list_files.append(filepath)
                continue
            if os.path.normpath(filepath) not in list_files:
                list_files.append(os.path.normpath(filepath))
        return sorted(list_files)

    def get_reference_files(self):
        """
        Helper function that return filtered
        and ordered list of all files paths Maya know about

        Returns:
            list: Ordered list of file paths
        """
        cmds.filePathEditor(refresh=True)
        tmp_files = list()
        files = cmds.file(q=True, list=True)
        for filename in files:
            if re.search(r"\{\".+", filename):
                # print("research: ", filename)
                continue
            # if not os.path.exists(filename): #managed by missing_file
            #     # add log warning
            #     continue
            if not self.remove_unloaded_ref(filename):
                continue
            if filename.endswith("}"):
                filename = filename.split("{")[0]
            tmp_files.append(os.path.normpath(filename))
        if RENDERER == "arnold":
            aiimages = self.get_aiimage_files()
            if aiimages:
                for path in aiimages:
                    if path not in tmp_files:
                        tmp_files.append(path)
        return sorted(tmp_files)

    def get_all_files(self):
        """
        Collect all files that Maya can detect in the scene, including specific cases for different renderers.
        Currently supports Renderman and Arnold renderer specifics.
        Should be also support better various OS by normalizing all the path

        Returns:
            list: A filtered list of file paths to include in the archive.

        Todos:
            Add more complete list of use cases for Renderman color spaces.
            Support for any similar case for others renderers
        """
        self.all_files = list()
        try:
            for current_file in self.get_reference_files():
                if current_file not in self.all_files:
                    self.all_files.append(current_file)
                if RENDERER == "arnold":
                    if current_file.endswith(".tx"):
                        continue
                    tx_files = glob(os.path.splitext(current_file)[0] + "*.tx")
                    if not tx_files:
                        continue
                    else:
                        for tx_file in tx_files:
                            if tx_file not in self.all_files:
                                self.all_files.append(tx_file)
                if RENDERER == "renderman":
                    if current_file.endswith(".tex"):
                        continue
                    tex_files = glob(os.path.splitext(current_file)[0] + "*.tex")
                    if not tex_files:
                        continue
                    else:
                        for tex_file in tex_files:
                            if tex_file not in self.all_files:
                                self.all_files.append(tex_file)
        except Exception:
            self.log_warning("No Textures files found")
        # self._copy_renderman_tex()

    def get_destination(self):
        if not self.ui:
            return self.destination

        if self.ui.destination:
            return self.ui.destination
        return SCENE_PATH

    def get_archive_name(self):
        if not self.ui:
            return self.job_name
        if self.ui.job_name:
            return str(self.ui.job_name)  # .replace(" ", "_")
        name = os.path.splitext(Utils.get_scene_name())[0]
        return str(UNSUPPORTED_CHARACTER.sub("_", name))

    @property
    def temp_folder(self):
        return self.__temp_folder

    @temp_folder.setter
    def temp_folder(self, temp):
        self.__temp_folder = temp

    def create_temp(self):
        """create a temporary folder at the scene location to manage some files"""
        temp_name = TEMP_PREFIXE + self.get_archive_name()
        temp_folder = os.path.join(self.get_destination(), temp_name)
        try:
            if os.path.exists(temp_folder):
                shutil.rmtree(temp_folder, ignore_errors=True)
            os.mkdir(temp_folder)
            os.mkdir(os.path.join(temp_folder, "scenes"))
            return temp_folder
        except OSError as error:
            return None

    def _remove_temp(self, temp_folder):
        """delete the temporary folder created at the beginning of the process"""
        try:
            if os.path.exists(temp_folder):
                shutil.rmtree(temp_folder, ignore_errors=True)
        except OSError as error:
            raise

    def _create_log_file(self, logs):
        log_path = os.path.join(self.temp_folder, "RANCHecker.log")
        try:
            log_path = os.path.join(self.temp_folder, "RANCHecker.log")
            # self.log_info("DEBUG LOG FILE: {}".format(log_path))
            with open(log_path, "w") as f:
                for log in self._write_log_file():
                    f.write(log + "\n")
                for log in logs:  # ajoute les WARNING, ERROR et INFO au fichier ranchecker.log
                    key = list(log.keys())[0]
                    f.write(key + " : " + log.get(key) + "\n")
            if os.path.normpath(log_path) not in self.all_files:
                self.all_files.append(os.path.normpath(log_path))
        except Exception as e:
            self.log_error("Could not write RANCHecker.log {}".format(e))

    def _create_ws_file(self, zip):
        if self.ui and not self.ui.is_running:
            self.zip.close()
            return
        ws_os = cmds.about(os=True)
        ws_path = Utils.get_workspace()
        # print("write ws: ", ws_path)
        if ws_os != "win64":
            ws_path = ":" + ws_path  # for MacOs and Linux
        try:
            with open(os.path.join(self.temp_folder, "ws.txt"), "w") as f:
                if ws_path:
                    f.write(ws_path)
                else:
                    f.write("C:\\Maya\\Currentjob")
            zip.write(os.path.join(self.temp_folder, "ws.txt"), "pref/" + "ws.txt")
        except:
            self.log_error("Could not write file: {}".format("ws.txt"))

    def _create_info_file(self, temp_path):
        try:
            with open(os.path.join(temp_path, "RANCHecker.info"), "w") as f:
                for i in self._write_info_file():
                    f.write(i)
        except:
            self.log_error("Could not write file: {}".format("RANCHecker.info"))

    def _create_command_line_file(self, temp_path):
        try:
            version = "Maya" + str(Utils.get_maya_version())
            with open(os.path.join(temp_path, "command_line.txt"), "w") as f:
                command_line = str(COMMAND_LINE).format(version) + " -scene {} ".format(
                    self.get_archive_name() + os.path.splitext(Utils.get_scene_name())[1]
                )
                f.write(command_line)
        except:
            self.log_error("Could not write file: {}".format("RANCHecker.info"))

    def _create_stillimage_file(self, temp_path):
        # animation
        if Utils.get_framerange_mode():
            return
        # still_images
        if Utils.get_stillimage_mode(qt_ui=self.ui) == "Single":
            self._create_singlenode_file(temp_path)
            if self.ui and self.ui.tiling_mode.isChecked():
                # warning due to refresh issues
                self.log_warning("Tiling render mode is activated, but will be ignored due to mismatch conditions")
        else:
            self._create_maxstrips_file(temp_path)

    def _create_singlenode_file(self, temp_path):
        try:
            with open(os.path.join(temp_path, "SingleNode.txt"), "w") as f:
                f.write("")
        except Exception as e:
            self.log_error("Could not write file: {} - {}".format("SingleNode.txt", e))

    def _create_maxstrips_file(self, temp_path):
        try:
            if self.ui.multibands_text.text() == "Auto Tiles" or self.ui.multibands.currentText() == "Auto Tiles":
                # Todo: fix refreshing issue with resolution
                print("max strip not necessary")
                if "MaxStrips.txt" in RANCH_ARCHIVE_FILES:
                    RANCH_ARCHIVE_FILES.remove("MaxStrips.txt")
                return
            with open(os.path.join(temp_path, "MaxStrips.txt"), "w") as f:
                f.write(self.ui.multibands.currentText().replace(" Tiles", ""))
            self.log_info("Multibands Render with [ {} ]".format(self.ui.multibands.currentText()))
        except Exception as e:
            self.log_error("Could not write file {} - {}".format("MaxStrips.txt", e))

    def _create_renderer_detected(self, temp_path):
        try:
            with open(os.path.join(temp_path, "renderer_detected.txt"), "w") as f:
                f.write(Utils.get_render_farm()[1])
        except:
            self.log_error("Could not write file: {}".format("renderer_detected.txt"))

    def _write_info_file(self):
        infos = list()
        infos.append(str(Utils.get_w_resolution()) + "x" + str(Utils.get_h_resolution()) + "\n")
        if cmds.getAttr("defaultRenderGlobals.animation"):
            infos.append(str(Utils.get_start_frame()) + "-" + str(Utils.get_end_frame()) + "\n")
        else:
            frame = int(cmds.currentTime(query=True))
            infos.append(str(frame) + "-" + str(frame) + "\n")
        infos.append(self.renderable_camera + "\n")
        infos.append(self.render_output + "\n")
        infos.append(Utils.get_render_farm()[0] + "\n")
        infos.append(self.renderable_scene)
        return infos

    def _write_log_file(self):
        logs = list()
        list_plugins = cmds.pluginInfo(query=True, pluginsInUse=True, version=True)
        logs.append("INFO : RANCHecker version | {}".format(__version__))
        logs.append("INFO : Maya version | {}".format(Utils.get_maya_version()))
        logs.append("INFO : Operating system | {}".format(Utils.get_operating_system()))
        logs.append("INFO : Output | {}".format(Utils.get_render_output()))
        logs.append("INFO : Render scene | {}".format(self.renderable_scene))
        logs.append("INFO : Render camera | {}".format(self.renderable_camera))
        logs.append("INFO : Farm | {}".format(Utils.get_render_farm()[0]))
        logs.append(
            "INFO : Frame range | {}".format(str(Utils.get_start_frame()) + "-" + str(Utils.get_end_frame()))
        )
        logs.append(
            "INFO : Resolution | {}".format(str(Utils.get_w_resolution()) + "x" + str(Utils.get_h_resolution()))
        )
        logs.append("INFO : List plugins | {}".format(list_plugins))
        return logs

    def change_ass_path(self, file_path):
        """
        Use read/write to replace paths in ASS files if Api doesn't work.
        """
        if file_path.endswith("ass.gz"):
            self.log_warning(".ass.gz not supported with this version of Arnold")
            return
        path = None
        try:
            with open(file_path, "r+") as f:
                read_file = f.readlines()
                for i, l in enumerate(read_file):
                    regex = re.compile(r"\sfilename (.*)")
                    match = regex.match(str(l))
                    if match:
                        _path = match.group(1).replace('"', "")
                        if os.path.exist(_path) and os.path.normpath(_path) not in self.all_files:
                            self.all_files.append(os.path.normpath)
                        if _path.endswith(".vdb"):
                            path = "C:/Maya/Currentjob/cache/vdb/" + os.path.basename(_path)
                        elif _path.endswith(".ies"):
                            path = "C:/Maya/Currentjob/IES/" + os.path.basename(_path)
                        elif _path.endswith(".abc"):
                            path = "C:/Maya/Currentjob/scenes/" + os.path.basename(_path)
                        elif _path.endswith((".ass", ".ass.gz")):
                            path = "C:/Maya/Currentjob/" + os.path.basename(_path)
                        elif _path.endswith(".usd"):
                            path = CACHE_PATH.replace("\\", "/") + os.path.basename(_path)
                        else:
                            path = os.path.normpath(SOURCEIMAGES_PATH + _path.split(":")[-1]).replace("\\", "/")
                        read_file[i] = " filename " + '"' + path + '"' + "\n"
                        f.seek(0)
                f.writelines(read_file)
        except:
            pass

    def replace_path_in_ass_ref(self, ass_path, sequence=False):
        """
        Use Arnold API to replace filepath by Ranch path in ASS files.
        https://help.autodesk.com/view/ARNOL/ENU/?guid=Arnold_API_REF_arnold_ref_index_html
        Args:
        - ass_path (str): the .ass file full path to query, ex: r"C:\\Users\\d.coco\\Documents\\maya\\projects\\default\\scenes\\Ass\\Assets_ref_dev.ass"
        - sequence (bool): add the logging for sequence, True if the file is the starting frame
        """
        print("Replacing paths for ", os.path.basename(ass_path))
        # function call encapsuled for arnold version switch
        # arnold_api_dict = {
        #     "current": {"load": AiSceneLoad, "write": AiSceneWrite, "mask": None},
        #     "deprecated": {"load": AiASSLoad, "write": AiASSWrite, "mask": AI_NODE_ALL},
        # }
        file_count = 0
        # self.log_info("Replacing paths for {}".format(os.path.basename(ass_path)))
        # Init Process
        AiBegin()

        # Load Ass File
        if not ARNOLD_VERSION.startswith(("2", "3")):
            arnold_api_dict = Utils.arnold_current_api()
            # AiSceneLoad(None,ass_path,None,)
        else:
            arnold_api_dict = Utils.arnold_legacy_api()
            # AiASSLoad(None, ass_path)

        rw_api_args = arnold_api_dict["universe"] + [ass_path, arnold_api_dict["mask"]]
        arnold_api_dict["load"](*rw_api_args)

        # Init parameter for ass file type, not working with AiASS
        # params = AiParamValueMap()
        # AiParamValueMapSetInt(params, "mask", AI_NODE_ALL)

        # Init parameters for OpenUSD file type
        # AiParamValueMapSetStr(params, "scope", "world")
        # AiParamValueMapSetStr(params, "Xform", "/world/mtl")

        # Iterate over all texture nodes
        iter_api_args = arnold_api_dict["universe"] + [AI_NODE_ALL]
        iter = AiUniverseGetNodeIterator(*iter_api_args)
        while not AiNodeIteratorFinished(iter):
            node = AiNodeIteratorGetNext(iter)

            # Check if the node is ASS options
            if AiNodeIs(node, "options"):
                AiNodeSetStr(node, "texture_searchpath", "C:/Maya/Currentjob/")
                AiNodeSetStr(node, "procedural_searchpath", "C:/Maya/Currentjob/")

            # Check if the node is an Image file
            if AiNodeIs(node, "image"):
                # Get the filename parameter value
                orig_tex_image_name = AiNodeGetStr(node, "name")
                orig_tex_filepath = AiNodeGetStr(node, "filename")
                print("Replacing texture filepath :: {} : {} ".format(orig_tex_image_name, orig_tex_filepath))
                if os.path.normpath(orig_tex_filepath) not in self.all_files:
                    self.all_files.append(os.path.normpath(orig_tex_filepath))

                # new_file_path = "/".join(("C:/Maya/Currentjob/sourceimages", os.path.basename(orig_tex_filepath)))
                new_tex_filepath = "".join((SOURCEIMAGES_PATH, os.path.normpath(orig_tex_filepath).split(":")[-1]))
                new_tex_filepath = new_tex_filepath.replace("\\", "/")
                AiNodeSetStr(node, "filename", new_tex_filepath)
                # print("Ranch path :: {} ".format(new_tex_filepath))
                file_count += 1

            # Check if the node is an aiStandardVolume
            if AiNodeIs(node, "volume"):
                # Get the filename parameter value
                orig_vdb_image_name = AiNodeGetStr(node, "dcc_name")
                orig_vdb_filepath = AiNodeGetStr(node, "filename")
                print("Replacing vdb filepath :: {} : {} ".format(orig_vdb_image_name, orig_vdb_filepath))

                if os.path.normpath(orig_vdb_filepath) not in self.all_files:
                    self.all_files.append(os.path.normpath(orig_vdb_filepath))

                new_vdb_filepath = "/".join(("C:/Maya/Currentjob/cache/vdb", os.path.basename(orig_vdb_filepath)))
                AiNodeSetStr(node, "filename", new_vdb_filepath)
                # print("Ranch path :: {} ".format(new_vdb_filepath))
                file_count += 1
            # Check if the node is an Alembic (gpu cache)
            if AiNodeIs(node, "alembic"):
                orig_abc_image_name = AiNodeGetStr(node, "dcc_name")
                orig_abc_filepath = AiNodeGetStr(node, "filename")
                print("Replacing abc filepath :: {} : {} ".format(orig_abc_image_name, orig_abc_filepath))

                if os.path.normpath(orig_abc_filepath) not in self.all_files:
                    self.all_files.append(os.path.normpath(orig_abc_filepath))

                new_abc_filepath = "/".join(("C:/Maya/Currentjob/scenes", os.path.basename(orig_abc_filepath)))
                AiNodeSetStr(node, "filename", new_abc_filepath)
                # print("Ranch path :: {} ".format(new_abc_filepath))
                file_count += 1

            # Check if the node is a Photometric light
            if AiNodeIs(node, "photometric_light"):
                orig_ies_image_name = AiNodeGetStr(node, "dcc_name")
                orig_ies_filepath = AiNodeGetStr(node, "filename")
                print("Replacing ies filepath :: {} : {} ".format(orig_ies_image_name, orig_ies_filepath))

                if os.path.normpath(orig_ies_filepath) not in self.all_files:
                    self.all_files.append(os.path.normpath(orig_ies_filepath))

                new_ies_filepath = "/".join(("C:/Maya/Currentjob/IES", os.path.basename(orig_ies_filepath)))
                AiNodeSetStr(node, "filename", new_ies_filepath)
                # print("Ranch path :: {} ".format(new_ies_filepath))
                file_count += 1

            if AiNodeIs(node, "usd"):
                orig_usd_filepath = AiNodeGetStr(node, "filename")
                print("Replacing ies filepath : {} ".format(orig_usd_filepath))
                if os.path.exists(orig_usd_filepath) and os.path.normpath(orig_usd_filepath) not in self.all_files:
                    self.all_files.append(os.path.normpath(orig_usd_filepath))
                    new_usd_filepath = CACHE_PATH.replace("\\", "/") + os.path.basename(orig_usd_filepath)
                    AiNodeSetStr(node, "filename", new_usd_filepath)
                    file_count += 1
                else:
                    self.log_warning("USD in ASS not found {}".format(orig_usd_filepath))

            if AiNodeIs(node, "procedural"):
                # Get the filename parameter value
                orig_ass_image_name = AiNodeGetStr(node, "name")
                orig_ass_filepath = AiNodeGetStr(node, "filename")
                print("replacing ass filepath :: {} : {} ".format(orig_ass_image_name, orig_ass_filepath))

                # add to standIns list for "_copy_arnold_ass"
                if orig_ass_filepath not in self.standin_path_list:
                    self.standin_path_list.append(orig_ass_filepath)
                new_ass_filepath = "/".join(("C:/Maya/Currentjob", os.path.basename(orig_ass_filepath)))
                AiNodeSetStr(node, "filename", new_ass_filepath)
                # print("Ranch path :: {} ".format(new_ass_filepath))
                file_count += 1

        start_frame = str(Utils.get_start_frame()).zfill(4)
        if start_frame in os.path.basename(ass_path):
            sequence = False
        if sequence == False and file_count != 0:
            self.log_info(
                "{} Paths replaced for {}".format(
                    file_count, os.path.basename(ass_path).replace(start_frame, "####")
                )
            )

        # Write to ASS File
        # AiSceneWrite(
        arnold_api_dict["write"](*rw_api_args)

        # Write to USDA file
        # AiSceneWrite(None, "\\".join((ass_project_path, "test.usda")), params)
        # AiParamValueMapDestroy(params)
        AiEnd()

    def _copy_ass_sequences(self, path_dir):
        """
        Copies .ass files from a provided directory to a temporary folder, updates embebbed filepaths, and
        adds the updated paths to a list.

        Search the specified directory for .ass files. Copies these files to a temporary folder if their name starts with the basename of the input path
        (excluding the same number of '#' characters from the end).
        After copying, it call a function to updates the filepath embbebed in the .ass.
        In case of any IO errors during copying, it logs the error and continues with the next file.

        Args:
            path_dir (str): The path of the directory from which to copy the .ass files.

        Returns:
            None
        """
        if not path_dir:
            return
        ass_seq = self.get_sequence_files(path_dir, "#")
        for ass in ass_seq:
            temp_path = os.path.join(self.temp_folder, os.path.basename(ass))

            try:
                shutil.copy2(ass, temp_path)
            except IOError:
                print("Failed to copy: " + ass + " to " + temp_path)
                continue

            # if not ARNOLD_VERSION.startswith("2"):
            self.replace_path_in_ass_ref(temp_path, sequence=True)
            # else:
            #     self.change_ass_path(temp_path)
            if temp_path not in self.all_files:
                self.all_files.append(temp_path)

    def _copy_arnold_ass(self):
        """
        Copies caches file used in Arnold AiStandIn nodes to a temporary folder and updates the embebbed filepaths.

        Search the caches file for Arnold Stand-in nodes.
        If the file format is .ass, it's copied to a temporary folder (self.temp_folder) and it call a function to update the filepath inside .ass.
        If "#" is in the path it call another function "_copy_ass_sequences" to process sequences.
        If the file format is .abc or .usd, its path is added to the list of all processed files.
        In case of any error during this process, the error message is printed to the console.

        Args:
            None

        Returns:
            None
        """
        # start_time_ass = time.time()
        try:
            if RENDERER == "arnold":
                standin_nodes = cmds.ls(et="aiStandIn")
                if not standin_nodes:
                    return
                for ass in standin_nodes:
                    ass_path = cmds.getAttr(ass + ".dso")
                    self.standin_path_list.append(ass_path)
                    # print(ass_path)
                for path in self.standin_path_list:
                    temp_ass = os.path.join(self.temp_folder, os.path.basename(path))
                    regex = re.compile(r"\d{4}.ass")
                    match = regex.findall(path)
                    # print("regex: ", match)
                    if path.endswith(".abc") or path.endswith(".usd"):
                        if os.path.normpath(path) not in self.all_files:
                            self.all_files.append(os.path.normpath(path))
                            continue
                    elif (
                        os.path.exists(path)
                        and path.endswith((".ass", ".ass.gz"))
                        and temp_ass not in self.all_files
                    ):
                        seq = False
                        shutil.copy2(path, self.temp_folder)
                        # if not ARNOLD_VERSION.startswith("2"):
                        if match and path.endswith(match[0]):
                            seq = True
                        self.replace_path_in_ass_ref(temp_ass, seq)
                        # else:
                        #     self.change_ass_path(temp_ass)
                        if temp_ass.endswith((".ass", ".ass.gz")):
                            self.all_files.append(temp_ass)
                    elif path.endswith(("#.ass", "#.azz.gz")):
                        self._copy_ass_sequences(path)
            # print("final standin list: ", self.standin_path_list)
            # print("Managing Ass in %s seconds" % (time.time() - start_time_ass))

        except Exception as e:
            print("Exception: ", e)

    def _copy_mash_cache(self):
        """
        Collect the Geometry cache files used in the scene and append to all_files list
        file formats : .mc, .mcx, .xml

        Read the XML file to get information about geometry cache type and format.
        If OneFile directly add to all_files
        Else, OneFilePerFrame, check the directory to find all the matching MCX/MC files
        """
        cache_files = cmds.ls(type="cacheFile")
        if not cache_files:
            return
        for cache in cache_files:
            try:
                if not cmds.getAttr(cache + ".enable"):
                    continue
                # collect xml
                cache_dir = cmds.getAttr(cache + ".cachePath")
                cache_name = cmds.getAttr(cache + ".cacheName")
                xml_path = os.path.normpath(os.path.join(cache_dir, cache_name + ".xml"))
                if xml_path not in self.all_files:
                    self.all_files.append(xml_path)
                # read xml information to collect mcx/mc
                xml_tree = ET.parse(xml_path)
                file_format = [child.attrib for child in xml_tree.getroot() if child.tag == "cacheType"]
                geo_format = "." + file_format[0]["Format"]
                geo_type = file_format[0]["Type"]
                if geo_type == "OneFile":
                    geo_cachepath = os.path.normpath(os.path.join(cache_dir, cache_name + geo_format))
                    if geo_cachepath not in self.all_files:
                        self.all_files.append(geo_cachepath)
                    continue
                # sequence: "OneFilePerFrame"
                for _file in os.listdir(cache_dir):
                    geo_cache = os.path.join(cache_dir, cache_name)
                    if not _file.startswith(geo_cache) and _file.endswith(geo_format):
                        continue
                    geo_cachepath = os.path.normpath(os.path.join(cache_dir, _file))
                    if geo_cachepath not in self.all_files:
                        self.all_files.append(os.path.normpath(_file))
            except Exception as e:
                self.log_warning("Could not read XML for geo cache {}".format(e))

    def check_paths(self, path):
        pass

    def _copy_alembic(self):
        """
        Check Node from type AlembicNode and append to all_files list
        """
        abc_nodes = cmds.ls(type="AlembicNode")
        if not abc_nodes:
            return
        for abc in abc_nodes:
            abc_path = cmds.getAttr(abc + ".abc_File")
            if os.path.normpath(abc_path) not in self.all_files:
                self.all_files.append(os.path.normpath(abc_path))

    def _replace_xgen_file(self, file):
        """
        Read an XGEN file and modify lines with paths and return all lines
        If the path is a cache file append to all_files list and replace with scene path
        If the path is for a mask IFF file replace drive by relative path

        Args:
            file : the full path of the XGEN file

        Returns:
            a list of all lines to write a new xgen file
        """
        list_line = list()
        query_word = "MapTextures"
        next_index = 0
        relative_path = "${PROJECT}sourceimages"
        # if file.endswith(".xgen"):
        with open(file, "r") as op_file:
            all_lines = op_file.readlines()
            for line in all_lines:
                orig_line = line
                next_index = all_lines.index(line, next_index) + 1
                line = line.strip(" \t\r\n")

                # set workspace, Todo: find a relative solution
                if line.startswith("xgProjectPath"):
                    split_line = line.split("\t")
                    path = split_line[-1]
                    line = orig_line.replace(path, "C:/Maya/Currentjob/")
                    print("project path: ", line)
                    list_line.append(line)
                    continue

                # guide Animation; todo check if "useCache" is true to avoid unnecessary size in vua
                if line.startswith("cacheFileName"):
                    split_line = line.split("\t")
                    path = split_line[-1]
                    # skip non-supported format and basic data
                    if not path.endswith((".abc", ".caf")) or path == "${DESC}/guides.abc":
                        list_line.append(orig_line)
                        continue
                    dirname = os.path.dirname(path)
                    line = orig_line.replace(dirname, "C:/Maya/Currentjob/scene")
                    print("cache line: ", line)
                    list_line.append(line)
                    continue

                # all the masks paths
                if query_word == "endAttrs":
                    query_word = "MapTextures"
                    list_line.append(orig_line)
                    continue
                if line.startswith(query_word):
                    next_line = all_lines[next_index]
                    query_word = next_line.strip(" \t\r\n")
                    split_line = line.split("\t")
                    path = split_line[-1]
                    if path == "MapTextures" or not split_line[-1].endswith(".iff"):
                        list_line.append(orig_line)
                        continue
                    print("Before: ", line)
                    line = orig_line.replace(path.split("/")[0], relative_path)
                    list_line.append(line)
                    print("After: ", line)
                else:
                    list_line.append(orig_line)

        return list_line

    def _copy_xgen_file(self):
        """
        Use xgenm api to find XGEN files and guide animation files.
        The path is append to all_files function
        """
        try:
            if xg.xgenProjectPath():
                xgen_dir = xg.xgenProjectPath()
                for desc in xg.descriptions():
                    palette = xg.palette(desc)  # palette = collection
                    # collect guide animation
                    objects = xg.objects(palette, desc, True)
                    for obj in objects:
                        if xg.getAttr("useCache", palette, desc, obj):
                            if xg.getAttr("useCache", palette, desc, obj) == "true":
                                guideanim_path = xg.getAttr("cacheFileName", palette, desc, obj)
                                if os.path.normpath(guideanim_path) not in self.all_files:
                                    self.all_files.append(os.path.normpath(guideanim_path))
                    # collect XGEN
                    xgen_file = cmds.getAttr(palette + ".xgFileName")
                    if not os.path.exists(xgen_file):
                        coll_dir = os.path.join(xgen_dir, "collections", palette)
                        for root, dirs, files in os.walk(coll_dir, topdown=True):
                            if not files:
                                # self.log_warning("No Xgen files found ")
                                continue
                            for filename in files:
                                full_path = os.path.join(root, filename)
                                # print("full_path", full_path)
                                xgen_path = full_path.split(xgen_dir)[1].replace("\\", "/")
                                print("Collected XGen files :", xgen_path)
                                if os.path.normpath(full_path) not in self.all_files:
                                    self.all_files.append(os.path.normpath(full_path))
                                # zip.write(full_path, "xgen" + xgen_path) #add arg zip to use it again

        except Exception as e:
            # TODO add error in RANCHecker logs
            print("xgen error ", e)

    def _copy_xgen_abc_file(self):
        """
        Append xgen spline cache (alembic format) in all_files list
        """
        written = set()
        working_dir = cmds.workspace(query=True, rootDirectory=True)
        try:
            xgen_spline = cmds.ls(type="xgmSplineCache")
            if not xgen_spline:
                return
            for x in xgen_spline:
                path = cmds.getAttr(x + ".fileName")

                if re.search("(:/|^/)", path):
                    # Cas chemin absolu
                    ws_path = path
                else:
                    # Cas chemin relatif
                    ws_path = os.path.join(working_dir, path)

                if not os.path.exists(ws_path):
                    self.log_warning("could not found abc in xgen spline cache in path : {}".format(path))
                    continue

                if ws_path in written:
                    continue
                if os.path.normpath(ws_path) not in self.all_files:
                    self.all_files.append(os.path.normpath(ws_path))
                written.add(ws_path)
        except Exception as e:
            self.log_warning("xgen error : {}".format(e))

    def _copy_yeti(self):
        """
        Collect Yeti files to the archive.

        This method is responsible for copying yeti files from their original
        location to a zip archive.

        Args:
            * self (Yeti): The Yeti object that owns this method.
        Returns:
            None
        Raises:
            Exception: If an error occurs during the file collect operation.
        """
        furs_list = list()
        if cmds.pluginInfo("pgYetiMaya", query=True, loaded=True):
            yeti_nodes = cmds.ls(et="pgYetiMaya")
            if not yeti_nodes:
                return
            for yeti_node in yeti_nodes:
                if cmds.getAttr(yeti_node + ".fileMode") == 0:  # No cache
                    print("No cache required for ", yeti_node)
                    continue
                yeticache_dir = os.path.dirname(cmds.getAttr(yeti_node + ".cacheFileName"))
                if yeticache_dir == "":
                    self.log_warning("Yeti Cache enabled for {} without path".format(yeti_node))
                    continue
                if not re.search("(:/|^/)", yeticache_dir):
                    # chemin relatif
                    working_dir = Utils.get_workspace()
                    yeticache_dir = os.path.join(working_dir, yeticache_dir)
                if not os.path.exists(yeticache_dir):  # due to os.listdir not handled by try/except
                    self.log_warning("Yeti Cache folder not found: {}".format(yeticache_dir))
                    continue
                for filename in os.listdir(yeticache_dir):
                    if filename.endswith(".fur"):
                        file_path = str(os.path.join(yeticache_dir, filename))
                        # print("file_path norm: ", os.path.normpath(file_path))
                        furs_list.append(os.path.normpath(file_path))
            for furs in furs_list:
                if furs not in self.all_files:
                    self.all_files.append(furs)

    def _copy_ies(self):
        """
        Collect the IES files used in the scene and append to all_files list
        """
        ies_path = list()
        if RENDERER == "arnold":
            ies_list = cmds.ls(type="aiPhotometricLight")
            for ies in ies_list:
                ies_path.append(cmds.getAttr(ies + ".aiFilename"))

        if RENDERER == "renderman":
            pxr_node_type = [
                "PxrCylinderLight",
                "PxrDiskLight",
                "PxrRectLight",
                "PxrSphereLight",
            ]
            for node_type in pxr_node_type:
                for ies in cmds.ls(type=node_type):
                    ies_path.append(cmds.getAttr(ies + ".iesProfile"))
                    # ies_path += cmds.getAttr(ies + ".iesProfile") #split each letter from path now?

        if RENDERER == "redshift":
            ies_list = cmds.ls(type="RedshiftIESLight")
            for ies in ies_list:
                ies_path.append(cmds.getAttr(ies + ".profile"))

        for path in ies_path:
            if os.path.normpath(path) not in self.all_files:
                self.all_files.append(os.path.normpath(path))

    def manage_proxy_files(self, path_to_lines):
        """
        Writes the absolute paths of texture to a file named "mypathoverrides.txt".

        Args:
            path_to_lines (list[str]): List of absolute or relative texture paths.
            proxy_path (str): The path of the proxy file. Its directory is used as the reference directory.
        """
        # Todo: relative path
        mypathoverrides = os.path.join(self.temp_folder, "mypathoverrides.txt")
        # with open(mypathoverrides, "w+") as file:
        lines = list()
        for path in path_to_lines:
            # collect proxy textures
            if os.path.normpath(path) not in self.all_files:
                self.all_files.append(os.path.normpath(path))
            # create mypathoverrides.txt
            if os.path.exists(path):
                dirname = os.path.dirname(path)
                replace_path = SOURCEIMAGES_PATH.replace("\\", "/") + dirname.split(":")[-1]
                line_to_write = '"{}" '.format(dirname) + '"{}"\n'.format(replace_path)
                if line_to_write not in lines:
                    lines.append(line_to_write)
        with open(mypathoverrides, "w") as file:
            for line in lines:
                file.write(line)

    def redshift_proxy(self):
        """
        Check if the proxy path is a sequence and if the proxy contains embedded textures.

        If Use Frame Extension is True, append the proxy files with the basename in all_files list
        For embedded textures Redshift API is used.

        Side Effects:
            Embedded path of Proxy created by old Redshift version could not be query (before V2.5)
            REDSHIFT_PATHOVERRIDE_FILE and REDSHIFT_PATHOVERRIDE_STRING on user side allow collect of files,
            but batch replace will failed due to wrong path in mypathoverride.txt

        """
        if not RENDERER == "redshift":
            return
        start_time = time.time()
        proxy_embbed_paths = list()
        try:
            for proxy_node in cmds.ls(type="RedshiftProxyMesh"):
                proxy_path = cmds.getAttr(
                    "{}.computedFileNamePattern".format(proxy_node)
                )  # path with ####, fileName with 0000
                if cmds.getAttr(proxy_node + ".useFrameExtension"):
                    proxy_basename = os.path.basename(proxy_path)
                    proxy_seq_path = [_file for _file in self.all_files if _file.endswith(proxy_path)]
                    if proxy_seq_path:
                        proxy_dir = os.path.dirname(proxy_seq_path[0])
                    else:
                        proxy_dir = os.path.dirname(proxy_path)
                    proxy_full_path = os.path.join(proxy_dir, proxy_basename)
                    proxy_sequence = self.get_sequence_files(proxy_full_path, "#")
                if proxy_sequence:
                    for proxy_file in proxy_sequence:
                        # path must have forward slash
                        proxy_file = proxy_file.replace("\\", "/")
                        print("seq to coll: ", proxy_file)
                        embedded = mel.eval('rsProxy -q -dependencies "%s"' % proxy_file)
                        proxy_embbed_paths.extend(
                            [
                                path if not proxy_embbed_paths else path
                                for path in embedded
                                if path not in proxy_embbed_paths
                            ]
                        )
                        if os.path.normpath(proxy_file) not in self.all_files:
                            self.all_files.append(os.path.normpath(proxy_file))
                else:
                    embedded = mel.eval('rsProxy -q -dependencies "%s"' % proxy_path)
                    proxy_embbed_paths.extend(
                        [
                            path if proxy_embbed_paths else path
                            for path in embedded
                            if path not in proxy_embbed_paths
                        ]
                    )
            print("::", proxy_embbed_paths)
            if proxy_embbed_paths:
                self.manage_proxy_files(proxy_embbed_paths)
                mypathoverrides = os.path.normpath(os.path.join(self.temp_folder, "mypathoverrides.txt"))
                if mypathoverrides not in self.all_files:
                    self.all_files.append(mypathoverrides)
                else:
                    print("No embedded Texture paths in ", proxy_node)
            print("proxy manage in %s seconds" % (time.time() - start_time))
        except Exception as e:
            exc_type, exc_obj, exc_tb = sys.exc_info()
            type_error = type(e).__name__
            self.log_warning("{} | {} line: {}".format(e, type_error, exc_tb.tb_lineno))
            pass

    # def _copy_substance_cfg(self, zip):
    #     if self.ui and not self.ui.is_running:
    #         self.zip.close()
    #         return
    #     if Utils.is_plugin_loaded("substancelink") or Utils.is_plugin_loaded("Substance"):
    #         substance_config_file = cmds.substanceGetCacheFolder()
    #         norm_path = os.path.normpath(substance_config_file)
    #         dir_file = os.path.dirname(norm_path)
    #         cfg_file_path = dir_file + "\\substance.cfg"
    #         cfg_normal_path = os.path.normpath(cfg_file_path)
    #         if os.path.exists(cfg_normal_path):
    #             zip.write(cfg_file_path, "pref\\" + os.path.basename(cfg_normal_path))

    def _copy_color_management(self, zip):
        if self.ui and not self.ui.is_running:
            utils.executeInMainThreadWithResult(self.zip.close)
            return
        path = cmds.colorManagementPrefs(query=True, configFilePath=True)
        if not path:
            return
        maya_version = str(Utils.get_maya_version())
        # Note : For OCIO engine V2 it's always the version the 2022 folder according to the doc:
        # (https://help.autodesk.com/view/MAYAUL/2024/ENU/?guid=GUID-2D9BBB9D-49F0-44C6-8BFF-B237FD8ED746)
        ocio_version = "2022"
        ocio_path = "C:\\Maya\\Maya{}\\resources\\OCIO-configs\\Maya{}-default\\config.ocio".format(
            maya_version, ocio_version
        )
        if str(path).startswith("<MAYA_RESOURCES>") and "2022" in str(path):
            ocio_path = "<MAYA_RESOURCES>/OCIO-configs/Maya{}-default/config.ocio".format(ocio_version)
        if str(path).startswith("<MAYA_RESOURCES>") and "Maya-legacy" in str(path):
            ocio_path = "<MAYA_RESOURCES>/OCIO-configs/Maya-legacy/config.ocio"

        color_management_path = os.path.join(self.temp_folder, "color_management.xml")
        cm_enabled = cmds.colorManagementPrefs(query=True, cmEnabled=True)
        if cm_enabled:
            ex = cmds.colorManagementPrefs(exportPolicy=color_management_path)
            tree = ET.parse(color_management_path)
            root = tree.getroot()
            for conf in root:
                if conf.attrib["name"] == "OCIOConfigFile":
                    conf.set("value", ocio_path)
            tree.write(color_management_path)
            zip.write(color_management_path, "pref/" + os.path.basename(color_management_path))

    def _copy_vdb(self):
        """
        Collect the VDB files used in the scene.
        Use a dictionary with render engines as key:
        'node_type': Node type
        'path_attr': Attribute of the path
        'sequence_query': Sequence query (attribute or token)
        """
        vdb_dict = {
            "arnold": {"node_type": "aiVolume", "path_attr": ".filename", "sequence_query": ".useFrameExtension"},
            "redshift": {
                "node_type": "RedshiftVolumeShape",
                "path_attr": ".fileName",
                "sequence_query": ".useFrameExtension",
            },
            "renderman": {
                "node_type": "OpenVDBRead",
                "path_attr": ".VdbFilePath",
                "sequence_query": ["<f>", "<f2>", "<f3>", "<f4>", "<f5>"],
            },
            "vray": {
                "node_type": "VRayVolumeGrid",
                "path_attr": ".inPath",
                "sequence_query": ".inMode",
            },  # check "#" or "inMode"=1 means static sequence
        }

        if RENDERER not in vdb_dict:
            return

        vdb_list = []
        vdb_volume = cmds.ls(et=vdb_dict[RENDERER]["node_type"])
        if not vdb_volume:
            return

        for vdb in vdb_volume:
            vdb_path, is_vdb_sequence = self._get_vdb_path_and_sequence(vdb, vdb_dict[RENDERER])
            self._collect_vdb_files(vdb_path, is_vdb_sequence, vdb_list)

        # print("vdb list: ", vdb_list) - For debug
        self.all_files += [vdb for vdb in vdb_list if vdb not in self.all_files]

    def _get_vdb_path_and_sequence(self, vdb, vdb_info):
        """Determine the VDB path and whether it is a sequence."""
        vdb_path = cmds.getAttr(vdb + vdb_info["path_attr"])
        is_vdb_sequence = False

        if RENDERER == "vray":

            is_vdb_sequence = (
                False
                if not cmds.getAttr(vdb + ".inCacheRange")
                else cmds.getAttr(vdb + vdb_info["sequence_query"]) == 1
            )
            if not is_vdb_sequence:
                vdb_path = cmds.getAttr(vdb + ".inPathResolved")
        elif RENDERER == "renderman":
            vdb_path = vdb_path.replace("<ws>", Utils.get_workspace())
            is_vdb_sequence = any(vdb_path[:-4].endswith(token) for token in vdb_info["sequence_query"])
        else:  # arnold, redshift
            is_vdb_sequence = cmds.getAttr(vdb + vdb_info["sequence_query"])

        return vdb_path, is_vdb_sequence

    def _collect_vdb_files(self, vdb_path, is_vdb_sequence, vdb_list):
        """Collect VDB files based on whether they are part of a sequence."""
        if is_vdb_sequence:
            dirname, vdbname = os.path.split(vdb_path)
            for vdb_name in os.listdir(dirname):
                if not vdb_name.endswith(".vdb") or (
                    "#" in vdbname and not vdb_name.startswith(vdbname.split("#")[0])
                ):
                    continue
                vdb_seq_path = os.path.join(dirname, vdb_name)
                if os.path.normpath(vdb_seq_path) not in vdb_list:
                    vdb_list.append(os.path.normpath(vdb_seq_path))
        else:
            if os.path.normpath(vdb_path) not in vdb_list:
                vdb_list.append(os.path.normpath(vdb_path))

    def _copy_bif(self):
        """_copy_bif query the .bif caches in the scene and add them to the list of files to collect"""
        bif_list = list()
        if cmds.pluginInfo("bifmeshio", query=True, loaded=True):
            bif_nodes = cmds.ls(typ="BifMeshImportNode")
            if not bif_nodes:
                pass
            else:
                for node in bif_nodes:
                    bif_dir = cmds.getAttr(node + ".bifMeshDirectory")
                    if not os.path.exists(bif_dir):
                        continue
                    for filename in os.listdir(bif_dir):
                        if filename.endswith(".bif"):
                            f_path = str(os.path.join(bif_dir, filename))
                            print("bif cache path: ", os.path.normpath(f_path))
                            bif_list.append(f_path)

        if cmds.pluginInfo("bifrostshellnode", query=True, loaded=True):
            bifrost_container = cmds.ls(et="bifrostContainer")
            if not bifrost_container:
                pass
            else:
                cont_warning = False
                containers_path = {
                    "liquid": [".enableLiquidCache", ".liquidCachePath", ".liquidCacheFileName"],
                    "liquidmesh": [".enableLiquidMeshCache", ".liquidmeshCachePath", ".liquidmeshCacheFileName"],
                    "solid": [".enableSolidCache", ".solidCachePath", ".solidCacheFileName"],
                    "aero": [".enableAeroCache", ".aeroCachePath", ".aeroCacheFileName"],
                }
                for cont in bifrost_container:
                    try:
                        # go to next element if container is disable or evaluation type is not "Property"
                        if not cmds.getAttr(cont + ".enable") or not cmds.getAttr(cont + ".evaluationType") == 1:
                            continue
                        for attribut in containers_path:
                            if cmds.getAttr(cont + containers_path[attribut][0]):
                                path = cmds.getAttr(cont + containers_path[attribut][1])
                                dir_name = cmds.getAttr(cont + containers_path[attribut][2])
                                content = os.walk(os.path.join(path, dir_name))
                                for root, _, files in content:
                                    for _file in files:
                                        bif_list.append(os.path.join(root, _file))
                                        cont_warning = True
                                        print("bif cache path: ", os.path.join(root, _file))
                    except Exception as e:
                        print("exception bifrost", e)
                        continue
                if cont_warning:
                    self.log_warning(
                        "Deprecated Bifrost Container caches will compute against dispatching process"
                    )

        for bifs in bif_list:
            if (
                os.path.normpath(bifs) not in self.all_files
            ):  # Ajouter un warning si dossier racine différent et dernier element du directory path egal
                self.all_files.append(os.path.normpath(bifs))

    def missing_file(self):
        """
        Check if a path exist on disk. If not display a warning.
        Ignore if item could not be file path.
        Ignore Renderman stats folder
        """
        # list_file_texture_name = cmds.filePathEditor(
        #     query=True, listFiles="", unresolved=False, attributeOnly=True
        # )
        for _path in self.all_files:
            # ignoring not file query
            if not any(slash in _path for slash in ["/", "\\"]) or _path == "":
                self.all_files.remove(_path)
                continue
            if RENDERER == "renderman":
                # ignore <ws>/stats/xx.x
                stats_folder = os.path.normpath(Utils.get_workspace() + "stats")
                basename = os.path.basename(_path).replace(".", "", 1)  # ex: 26.2 = 262
                if os.path.normpath(_path).startswith(stats_folder) and basename.isdigit():
                    self.all_files.remove(_path)
                    continue
            # warning
            if not os.path.exists(_path) and os.path.dirname(_path) != self.temp_folder:
                self.log_warning("Missing assets [ {} ] not found".format(_path))
                self.all_files.remove(_path)
                continue

    def _zip_files(self, file, zip):
        if self.ui and not self.ui.is_running:
            return
        # print("Check to zip: ", file)
        file_name = os.path.basename(file)
        if file_name in set(RANCH_ARCHIVE_FILES):
            zip.write(file, os.path.basename(file))
        elif file_name.lower().endswith((".mcj", ".ini", ".ocio")):
            return
        elif file.lower().endswith(
            (
                ".als",
                ".bmp",
                ".cin",
                ".dds",
                ".dpx",
                ".eps",
                ".exr",
                ".gif",
                ".hdr",
                ".iff",
                ".jpeg",
                ".jpg",
                ".m4v",
                ".mov",
                ".mp4",
                ".pic",
                ".png",
                ".psd",
                ".ptex",
                ".rla",
                ".rpf",
                ".sgi",
                ".svg",
                ".tex",
                ".tga",
                ".tif",
                ".tiff",
                ".tx",
                ".vrimg",
                ".webp",
                ".xpm",
                ".yuv",
            )
        ):
            if sys.platform == "win32":
                # Todo: Change this to not include the dirname path, and take in account hashable path
                norm_file_name = file.split(":")[1]
                zip.write(file, "sourceimages/" + norm_file_name)
            else:
                zip.write(file, "sourceimages/" + file)
        elif file_name.endswith((".mb", ".ma")):
            if file_name == Utils.get_scene_name() and file == os.path.normpath(
                cmds.file(query=True, sceneName=True)
            ):
                # TODO get maya file extension
                zip.write(
                    file,
                    os.path.join(
                        "scenes",
                        self.get_archive_name() + os.path.splitext(file_name)[1],
                    ),
                )
            else:
                zip.write(file, "scenes/" + file_name)
        elif file_name.endswith(".txt"):
            zip.write(file, file_name)
        elif file_name.lower().endswith((".abc", ".fbx")):
            zip.write(file, "scenes/" + file_name)
        elif file_name.endswith(".obj"):
            zip.write(file, "scenes/" + file_name)
            mtl_file = file.replace(".obj", ".mtl")
            if os.path.exists(mtl_file):
                file_name = os.path.basename(mtl_file)
                zip.write(mtl_file, "scenes/" + file_name)
        elif file_name.endswith(".wav"):
            zip.write(file, "sound/" + file_name)
        elif file_name.endswith(".avi"):
            zip.write(file, "movies/" + file_name)
        elif file_name.endswith(".bif"):
            bif_liquid = os.path.dirname(file)
            bif_dir = os.path.basename(bif_liquid)
            lp = os.path.basename(os.path.dirname(bif_liquid))
            zip.write(
                file,
                "cache/bifrost/" + os.path.join(lp, bif_dir, os.path.basename(file)),
            )
        elif file_name.endswith((".mc", ".mcx", ".xml")):
            zip.write(file, "cache/nCache/" + os.path.basename(file))
        elif file_name.endswith((".fur", ".grm")):
            zip.write(file, "cache/" + file_name)
        elif file_name.lower().endswith(".vdb"):
            zip.write(file, "cache/vdb/" + file_name)
        elif file_name.lower().endswith((".usd", ".usda", ".usdc", ".usdz")):  # Todo: change for scene location
            if file_name.endswith(".usdz"):
                self.log_warning("Zipped USD are not supported")
            zip.write(file, "cache/" + file_name)
        elif file_name.lower().endswith(".ies"):
            zip.write(file, "IES/" + file_name)
        elif file_name.endswith((".ass", ".ass.gz")):
            # zip.write(file, "scenes/" + file_name)
            zip.write(file, file_name)
        elif file_name.endswith(".rs"):
            zip.write(file, "proxy/" + file_name)
        elif file_name.endswith(".xgen"):
            xgen_lines = self._replace_xgen_file(file)
            # to match jobname
            if file_name.split("__")[0] is not self.get_archive_name():
                self.log_warning("use of Job name with xgen will result on missing fur, consult our doc")
            #     scn, coll = file_name.split("__")
            #     file_name = self.get_archive_name() + "__" + coll
            xg_temp = os.path.join(self.temp_folder, "scenes", file_name)
            with open(xg_temp, "w") as xgen_file:
                for current_line in xgen_lines:
                    xgen_file.write(current_line)
            zip.write(xg_temp, "scenes/" + os.path.basename(file_name))

            alembic_xgen = file.replace(".xgen", ".abc")
            if not os.path.exists(alembic_xgen):
                msg = (
                    "Please export xgen for batch rendering in .abc file, " "for more information consult our doc"
                )
                self.log_warning("can't found .abc file for xgen cache")
                self.log_info("Please export xgen for batch rendering in .abc file")
                self.message_emit.emit(msg)
            else:
                zip.write(alembic_xgen, "scenes/" + file_name.replace(".xgen", ".abc"))
        elif file_name.endswith((".ptx", ".xuv", ".xpd", ".xgc")):
            # print("not to zip: ", file)
            file_split = file.split("\\xgen\\collections")
            print("to zip: ", file_split[-1])
            if sys.platform == "win32":
                # Todo: Change this to not include the dirname path, and take in account hashable path
                norm_file_name = file.split(":")[1]
                zip.write(file, "xgen/collections/" + file_split[-1])
            else:
                zip.write(file, "xgen/collections/" + file_split[-1])
        else:
            self.log_warning("unknown extension for {}, contact the support for details".format(file_name))

    def _create_archive(self):
        self.vua_archive_path = os.path.join(self.get_destination(), self.get_archive_name() + ".vua")
        if self.ui and not self.ui.is_running:
            utils.executeInMainThreadWithResult(self.zip.close)
        if os.path.exists(self.vua_archive_path):
            utils.executeInMainThreadWithResult(lambda: shutil.rmtree(self.vua_archive_path, ignore_errors=True))

        utils.executeInMainThreadWithResult(self.clean_logs)

        utils.executeInMainThreadWithResult(lambda: cmds.filePathEditor(refresh=True))
        utils.executeInMainThreadWithResult(self.__set_ranch_archive_files)
        utils.executeInMainThreadWithResult(lambda: self._create_info_file(self.temp_folder))
        utils.executeInMainThreadWithResult(lambda: self._create_stillimage_file(self.temp_folder))
        utils.executeInMainThreadWithResult(lambda: self._create_renderer_detected(self.temp_folder))
        utils.executeInMainThreadWithResult(lambda: self._create_command_line_file(self.temp_folder))
        utils.executeInMainThreadWithResult(self._copy_bif)
        utils.executeInMainThreadWithResult(self._copy_mash_cache)
        utils.executeInMainThreadWithResult(self._copy_arnold_ass)
        utils.executeInMainThreadWithResult(self._copy_ies)
        utils.executeInMainThreadWithResult(self._copy_alembic)  # already query by maya file list
        utils.executeInMainThreadWithResult(self._copy_xgen_abc_file)
        utils.executeInMainThreadWithResult(self._copy_xgen_file)
        utils.executeInMainThreadWithResult(self._copy_yeti)
        utils.executeInMainThreadWithResult(self._copy_vdb)
        utils.executeInMainThreadWithResult(self.redshift_proxy)
        utils.executeInMainThreadWithResult(self.missing_file)
        try:
            self.zip = utils.executeInMainThreadWithResult(
                lambda: zipfile.ZipFile(self.vua_archive_path, "w", zipfile.ZIP_DEFLATED, allowZip64=True)
            )
            # self._copy_substance_cfg(self.zip) # TODO : A refaire

            utils.executeInMainThreadWithResult(lambda: self._copy_color_management(self.zip))
            utils.executeInMainThreadWithResult(lambda: self._create_ws_file(self.zip))
            utils.executeInMainThreadWithResult(lambda: self._create_log_file(self._logs))
            self.progress_range.emit(len(self.all_files) - 1)
            for index, file in enumerate(self.all_files):
                if os.path.exists(file):
                    utils.executeInMainThreadWithResult(lambda file=file: self._zip_files(file, self.zip))
                self.progress_value_text.emit(str(os.path.basename(file)), index)

            utils.executeInMainThreadWithResult(self.zip.close)
        except Exception as e:
            message = "VUA is opened could not overwrite it, \n close zip manager or use jobname"
            if type(e).__name__ == "PermissionError":
                utils.executeInMainThreadWithResult(
                    lambda: cmds.confirmDialog(title="RANCHecker", m=message, button=["Ok"])
                )
            self.ui.on_stop()
            self.log_error("{}".format(e))

        utils.executeInMainThreadWithResult(lambda: self._remove_temp(self.temp_folder))
        cmds.file(modified=0)
        self._remove_temp(self.temp_folder)

    def __set_ranch_archive_files(self):
        try:
            for file in set(RANCH_ARCHIVE_FILES):
                rc_path = os.path.join(self.temp_folder, file)
                if rc_path not in self.all_files:
                    self.all_files.append(rc_path)
        except Exception as e:
            print(e)
            pass

    def check_scene_name(self):
        maya_scene = mel.eval("file -q -ns")
        if self.ui.job_name != "":  # and not UNSUPPORTED_CHARACTER.search(self.ui.job_name):
            print("Job Name: {}".format(self.ui.job_name))
            return True
        elif not UNSUPPORTED_CHARACTER.search(maya_scene):
            print("Scene Name: {}".format(maya_scene))
            return True
        else:
            print("unsupported character detected: ", maya_scene)
            message = " ERROR: Special characters detected in scene name, renamed it"
            self.log_error(message)
            popup = cmds.confirmDialog(title="RANCHecker", m=message, button=["Ok"])
            if popup == "Ok":
                # self.on_stop_process()
                self.ui.on_stop()
            return False

    def save_scene(self):
        # Hack to set the scene dirty on scene save
        # cmds.file( modified=1 )
        if self.ui and not self.ui.job_name:
            if len(Utils.get_scene_name()) > 23:
                self.log_error("Scene name is too long, max 20 char")
                self.stopped = True

        self.saved = cmds.file(q=True, anyModified=True)
        if self.saved:
            save_scene_state = utils.executeInMainThreadWithResult(
                lambda: cmds.confirmDialog(
                    title="Save Scene ?",
                    m="Scene must be saved to create archive",
                    button=["Save", "Cancel"],
                )
            )
            if save_scene_state == "Cancel":
                self.log_error(
                    "Could not create archive, scene not saved",
                )
                self.ui.on_stop()
                # self.on_stop_process()
                return False
            elif (
                utils.executeInMainThreadWithResult(lambda: cmds.file(q=True, shn=True, sn=True)) == ""
                and save_scene_state == "Save"
            ):
                self.stopped = True

            elif save_scene_state == "Save" and cmds.file(q=True, shn=True, sn=True) == "":
                self.log_error(
                    "Scene was never saved, please save first and restart",
                )
                self.stopped = True
                return False

            elif save_scene_state == "Save":
                utils.executeInMainThreadWithResult(lambda: cmds.file(save=True, force=True))
        self.renderable_camera = Utils.get_render_camera()
        if self.renderable_camera is None:
            self.log_error("No Camera set in the Render Settings")
            return False
        self.renderable_scene = (
            self.get_archive_name() + os.path.splitext(Utils.get_scene_name())[1]
        )  # TODO check if maya ascii file # Utils.get_scene_name()
        self.render_output = Utils.get_render_output()
        self.get_all_files()
        self.temp_folder = self.create_temp()

        if self.temp_folder:
            try:
                short_scene_name = cmds.file(query=True, shortName=True, sceneName=True, force=1)
            except:
                self.log_error("failed to create temp folder ")
        else:
            self.log_error("failed to create temp folder ")
        return True

    def run(self):
        self.archive_creation_time = time.time()
        if utils.executeInMainThreadWithResult(lambda: self.check_scene_name()):
            self._create_archive()

    def on_stop_process(self):
        self.quit()
        self._remove_temp(self.temp_folder)


class Archive(QObject):
    def __init__(self):
        self.__is_running = False

    @property
    def is_running(self):
        return self.__is_running

    @is_running.setter
    def is_running(self, value):
        self.__is_running = value

    def create_archive(self):
        self.btn_archive = QPushButton("Prepare Project")
        self.btn_stop = QPushButton("Stop")
        self.btn_archive.setMinimumSize(QSize(100, 35))
        self.btn_stop.setMinimumSize(QSize(100, 35))
        self.btn_stop.setStyleSheet("background-color: rgb(220, 69, 85);")
        self.archive_submit_layout.addWidget(self.btn_stop)
        self.archive_submit_layout.addWidget(self.btn_archive)

    def archive_visibility(self, is_running=False):
        self.btn_stop.setVisible(is_running)
        self.btn_archive.setHidden(is_running)

    def _run_process(self):
        if self.is_running or self.stopped:
            return
        self.progress_bar.setValue(0)
        self.status.setText("In progress ...")
        self.is_running = True
        self.archive_visibility(self.is_running)
        # TODO: check if VUA open before save function, currently after Worker.missing file
        if self.worker.save_scene():
            self.worker.start()
        else:
            self.is_running = False
            self.archive_visibility(self.is_running)
            self.worker.on_stop_process()


class Submit(Api):
    def __init__(self):
        pass
        # Api.__init__(self)

    def create_submit(self):
        self.btn_submit = QPushButton("Upload Project")
        self.btn_submit.setMinimumSize(QSize(100, 35))
        self.archive_submit_layout.addWidget(self.btn_submit)

    def submit_visibility(self, visibility):
        self.btn_submit.setEnabled(visibility)

    def submit(self, renderer_id, priority_id, skip_finished_email, layer, index=1):
        camera = {}
        file_name = self.worker.get_archive_name() + ".vua"
        if layer:
            if cmds.getAttr("{}.renderable".format(layer)):
                render_layer = {"layer": layer}
                layer_file_name = (
                    self._rename_for_layer_submit(self.worker.get_archive_name(), layer, index) + ".vua"
                )
            else:
                self.worker.log_error("Please enable at least one layer or masterLayer!")

        vua_archive_path = os.path.join(self.get_destination(), file_name)

        data = {
            "method": "create_and_upload",
            "sweb": {
                "renderer_id": str(renderer_id),
                "priority_id": str(priority_id),
                "disclaimer": True,
                "_optional_parameters_see": "",
                "skip_finished_email": skip_finished_email,
                "size": {},
                "frames": self.specific_frames,
                "camera": camera,
                "layer": render_layer,
            },
            "upload": {
                "file_path": vua_archive_path,
                "filename": layer_file_name,
            },
            "download": {},
        }

        resp = self.send_to_ranchsync(data)
        if "error" in resp:
            self.worker.log_error("status : {}".format(resp["error"]))
            return
        if resp["status"] == "uploading":
            self.worker.log_info("Job id [ {} ]".format(resp["id"]))
            self.worker.log_info("Job name [ {} ]".format(resp["filename"]))
            self.worker.log_success("Job start uploading, Please check RANCHSync uploads tab")
        else:
            self.worker.log_warning("Failed to upload")

    def send_to_ranchsync(self, data):
        def recvall(sock):
            BUFF_SIZE = 4096  # 4 KiB
            data = b""
            while True:
                part = sock.recv(BUFF_SIZE)
                data += part
                if len(part) < BUFF_SIZE:
                    # either 0 or end of data
                    break
            return data

        s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        try:
            s.connect((HOST, PORT))
            json_data = json.dumps(data)
            try:
                s.sendall(bytes(json_data))
            except:
                s.sendall(bytes(json_data, encoding="utf-8"))
            b = recvall(s)
            try:
                return json.loads(str(b, "utf-8"))
            except:
                return json.loads(str(b))
        except Exception as e:
            urlLink = '<a href="https://www.ranchcomputing.com/en/retrieve-the-rendered-frames"> <font face=verdana color=cyan> Download RANCHSync</font> </a>'
            message = "Please run RANCHSync to upload your project (recommended).<br>{}".format(urlLink)
            QMessageBox.information(self, "RANCHecker", message)
            return False
        finally:
            s.close()


class RendererTab(Api):
    def __init__(self):
        pass

    def create_renderer(self):
        renderer_widgets = QWidget()
        h_text = QLabel("  The following renderer supported in the current Maya version:")
        h_text.setStyleSheet("font-size: 11pt;")
        renderer_lyout = QVBoxLayout()
        renderer_lyout.setContentsMargins(0, 0, 0, 0)

        self.renderer_view = QListWidget()
        self.renderer_view.setContentsMargins(0, 10, 0, 0)
        self.renderer_view.setViewMode(QListView.IconMode)
        self.renderer_view.setSelectionMode(QListWidget.NoSelection)
        renderer_lyout.addWidget(h_text)
        renderer_lyout.addWidget(self.renderer_view)
        renderer_widgets.setLayout(renderer_lyout)
        self.stack_view.addWidget(renderer_widgets)

    def _clear_renderer_view(self):
        self.renderer_view.clear()

    def set_gallery_item(self):
        self._clear_renderer_view()
        renderer = self.get_support_renderer()
        if renderer:
            for render in renderer:
                if render["name"] == "Arnold GPU" or render["name"] == "V-Ray GPU":
                    continue
                elif render["name"] == "Arnold":
                    __name = "arnold"
                elif render["name"] == "RenderMan":
                    __name = "renderman"
                elif render["name"] == "Maxwell":
                    __name = "maxwell"
                elif render["name"] == "V-Ray":
                    __name = "v-ray"
                elif render["name"] == "Redshift":
                    __name = "redshift"
                elif render["name"] == "Maya Software":
                    __name = "maya"
                item = QListWidgetItem()
                if not render["version"]:
                    item.setText("Maya Software")
                else:
                    item.setText(render["version"])
                url_image = ICON_RENDERER_URL + "{}.png".format(__name)
                req = Request(url_image)
                res = urlopen(req)
                image_data = res.read()
                image = QImage()
                image.loadFromData(image_data)
                item.setIcon(QIcon(QPixmap(image)))
                item.setSizeHint(QSize(80, 80))
                self.renderer_view.addItem(item)


class PluginsTab(Api):
    def __init__(self):
        pass

    def create_plugins(self):
        plugins_widgets = QWidget()
        plugins_lyout = QVBoxLayout()
        plugins_lyout.setContentsMargins(0, 0, 0, 0)
        h_text = QLabel("  The following plugins supported in the current Maya version:")
        h_text.setStyleSheet("font-size: 11pt;")
        self.plugins_view = QListWidget()
        self.plugins_view.setSpacing(1)
        plugins_lyout.addWidget(h_text)
        plugins_lyout.addWidget(self.plugins_view)
        plugins_widgets.setLayout(plugins_lyout)
        self.stack_view.addWidget(plugins_widgets)

    def _clear_plugins_view(self):
        self.plugins_view.clear()

    def set_plugins_item(self):
        self._clear_plugins_view()
        plugins = self.get_support_plugins()
        if plugins:
            for plugin in plugins:
                w = self._set_widget(plugin["name"], plugin["version"])
                item = QListWidgetItem()
                item.setSizeHint(QSize(20, 20))
                self.plugins_view.addItem(item)
                self.plugins_view.setItemWidget(item, w)


class InfosTab(QObject):
    def __init__(self):
        pass

    def _clear_infos_view(self):
        self.infos_view.clear()

    def create_infos(self):
        infos_widgets = QWidget()
        infos_lyout = QVBoxLayout()
        infos_lyout.setContentsMargins(0, 0, 0, 0)
        self.infos_view = QListWidget()
        self.infos_view.setSpacing(1)
        infos_lyout.addWidget(self.infos_view)
        infos_widgets.setLayout(infos_lyout)
        self.stack_view.addWidget(infos_widgets)

    def set_scene_info_item(self, msg):
        self._clear_infos_view()
        for key, value in msg.items():
            w = self._set_widget(key, value)
            item = QListWidgetItem()
            item.setSizeHint(QSize(20, 20))
            self.infos_view.addItem(item)
            self.infos_view.setItemWidget(item, w)


class Quick_check(QObject):
    """
    Quick_check is a utility class designed to perform various checks on a Maya scene
    and communicate the results back to the Worker class using signals.

    Attributes:
        worker (Worker): An optional reference to the Worker instance that uses this Quick_check instance.
        uploaded_layers (list): A list to store layers that have been uploaded.

    Signals:
        check_completed (str): Emitted when a check is completed, carrying a message about the result.

    Methods:
        some_check_method(): Performs a specific check and emits a signal with the result.
    """

    check_completed = Signal(str)

    def __init__(self, worker=None):
        super(Quick_check, self).__init__()
        self.worker = worker
        self.uploaded_layers = list()

    def _check_unsupported_engine(self):
        supported = Utils.list_api_renderer()
        renderer = Utils.get_current_renderer()
        if renderer.capitalize() not in supported:
            self.worker.log_error("Unsupported render engine {}".format(renderer))
            self.worker.ui.on_stop()

    def _check_rman_xpu(self):
        if RENDERER != "renderman" or (
            RENDERER == "renderman" and cmds.getAttr("rmanGlobals.renderVariant") == "ris"
        ):
            return
        if cmds.getAttr("rmanGlobals.xpuMode1") == 1:
            self.worker.log_warning("XPU Mode: GPU enable, but will be ignore as CPU farm does not have GPU")
        if cmds.getAttr("rmanGlobals.xpuMode0") == 0:
            self.worker.log_error("XPU Mode without CPU is not supported yet, please enable XPU Mode CPU")
            self.worker.ui.on_stop()

    def _arnold_api_check_ass(self, ass_path, parameter="All", node_module="Node"):
        """
        Use Arnold API to query ASS files nodes.
        https://help.autodesk.com/view/ARNOL/ENU/?guid=Arnold_API_REF_arnold_ref_index_html
        deprecated: https://help.autodesk.com/view/ARNOL/ENU/?guid=Arnold_API_REF_group_ai_dotass_html

        Args:
        - ass_path (str): the .ass file full path to query
        - parameter (str): the dictionary key to know the API list to parse
        - node_module (str): the arnold iterator to use, value can be "Node" or "NodeEntry"

        Return:
            list(str) elements of specify parameters

        """
        try:
            node_mask = {
                "All": AI_NODE_ALL,
                "Light": AI_NODE_LIGHT,
                "Shader": AI_NODE_SHADER,
                "Volume": AI_NODE_SHAPE_VOLUME,
            }

            AiBegin()

            # Load Ass File
            if not ARNOLD_VERSION.startswith(("2", "3")):
                arnold_api_dict = Utils.arnold_current_api()
                # AiSceneLoad(None,ass_path,None,)
            else:
                arnold_api_dict = Utils.arnold_legacy_api()
                # AiASSLoad(None, ass_path)

            rw_api_args = arnold_api_dict["universe"] + [ass_path, arnold_api_dict["mask"]]
            arnold_api_dict["load"](*rw_api_args)

            # Iterate over all texture nodes

            elements_list = list()

            if node_module == "Node":
                iter_api_args = arnold_api_dict["universe"] + [node_mask[parameter]]
                iter = AiUniverseGetNodeIterator(*iter_api_args)
                while not AiNodeIteratorFinished(iter):
                    node = AiNodeIteratorGetNext(iter)
                    node_name = AiNodeGetStr(node, "dcc_name")  # Node Name ex:"aiSkyDomeLightShape1"
                    # print("NAME: ", node_name)
                    if node_name != "":
                        elements_list.append(node_name)
            elif node_module == "NodeEntry":
                iter_entry = AiUniverseGetNodeEntryIterator(node_mask[parameter])
                while not AiNodeEntryIteratorFinished(iter_entry):
                    node = AiNodeEntryIteratorGetNext(iter_entry)
                    node_type = AiNodeEntryGetTypeName(node)  # Type groupe ex: "light"
                    # print("TYPE:", node_type)
                    node_name = AiNodeEntryGetName(node)  # Type name ex:"skydome_light"
                    # print("NAME:", node_name)
                    if node_name != "":
                        elements_list.append(node_name)

            AiEnd()

            return elements_list  # list element of specify parameter
        except:
            self.worker.log_warning("Failed to retrieve information from Arnold API, some content may be missing")

    def _check_is_light(self):
        """_check_is_light Return a warning if no lights is found in any supported renderer"""
        # TODO: check if light are embebbed in caches(.rs,.vrmesh)
        agreed_lights = set()
        collected_light = list()
        lights_list = {
            "mayaSoftware": [
                "directionalLight",
                "pointLight",
                "spotLight",
                "areaLight",
                "volumeLight",
                "ambientLight",
            ],
            "arnold": ["aiAreaLight", "aiSkyDomeLight", "aiPhotometricLight", "aiMeshLight"],
            "redshift": [
                "RedshiftPortalLight",
                "RedshiftPhysicalLight",
                "RedshiftIESLight",
                "RedshiftDomeLight",
                "RedshiftPhysicalSky",
            ],
            "renderman": [
                "PxrRectLight",
                "PxrDiskLight",
                "PxrDistantLight",
                "PxrSphereLight",
                "PxrCylinderLight",
                "PxrAovLight",
                "PxrDomeLight",
                "PxrPortalLight",
                "PxrEnvDayLight",
            ],
            "vray": [
                "VRayLightDomeShape",
                "VRayLightRectShape",
                "VRayLightSphereShape",
                "VRayLightMesh",
                "VRayLightIESShape",
                "VRaySunShape",
            ],
        }
        RENDERER = cmds.getAttr("defaultRenderGlobals.currentRenderer")  # init again for change
        if RENDERER in lights_list:
            print("Render Engine: {}".format(RENDERER.capitalize()))
            # Get a set of all unique lights for the current renderer and mayaSoftware
            agreed_lights = lights_list["mayaSoftware"] + lights_list[RENDERER]
            # print("agreed set: ", agreed_lights)
            # Collect all lights of the types specified in unique_lights
            collected_light = [
                light for light_type in agreed_lights for light in cmds.ls(type=light_type) if light
            ]
            if collected_light:
                return
            if not RENDERER == "arnold":
                self.worker.log_warning("No light in the scene, render will be black.")
                return
            else:
                try:
                    ass_light = False
                    standin_nodes = cmds.ls(et="aiStandIn")
                    for ass in standin_nodes:
                        if not ass.endswith(("ass", ".ass.gz")):
                            continue
                        ass_path = cmds.getAttr(ass + ".dso")
                        ass_light = self._arnold_api_check_ass(ass_path, parameter="Light")
                        collected_light.extend(ass_light)
                except Exception as e:
                    print(e)
                if not collected_light:
                    self.worker.log_warning("No light in the scene, render will be black.")
            # print(collected_light)  # Optional: Print the list of collected lights
            return

    def _check_jobname_long(self):
        if len(self.JOB_NAME) > 20:
            self.worker.log_error("The job name is too long (max 20 chars)")

    def _check_stillimage(self):
        """_check_stillimage
        check if the scene render a single frame and add SingleNode.txt or MaxStrips.txt to RANCH files list
        Returns:
            _type_: Bool
        """
        # reset to default RANCH_ARCHIVE_FILE list
        RANCH_ARCHIVE_FILES[:] = [
            file for file in RANCH_ARCHIVE_FILES if not file in ["MaxStrips.txt", "SingleNode.txt"]
        ]
        if Utils.get_framerange_mode():
            self.worker.log_info(
                "Render Animation for frames [ {} - {} ]".format(Utils.get_start_frame(), Utils.get_end_frame())
            )
        else:
            self.worker.log_info(
                "Render Still image in frame [ {} ]".format(str(int(cmds.currentTime(query=True))))
            )
            if Utils.get_stillimage_mode(qt_ui=self) == "Multi":
                RANCH_ARCHIVE_FILES.append("MaxStrips.txt")  # not for all tile
            else:
                RANCH_ARCHIVE_FILES.append("SingleNode.txt")
        return True

    def _check_arnold_tx_file(self):
        if cmds.getAttr("defaultRenderGlobals.currentRenderer") == "arnold":
            if cmds.getAttr("defaultArnoldRenderOptions.autotx") == 1:
                self.worker.log_info("Please convert texture(s) to [.tx] for a faster rendering.")

    def _rename_for_layer_submit(self, scene_name, layer, index=1):
        if not layer:
            return ""
        if len(scene_name) > 10:
            scene_name = scene_name[:10]
        if layer.startswith("rs_"):
            layer = layer[3:]
            if len(layer) > 10:
                layer = layer[:7] + str(index) + "_" + scene_name
            else:
                layer = layer + "_" + scene_name
        if layer == "defaultRenderLayer":
            layer = "ML_" + scene_name
        return layer

    def _check_legacy_layer(self):
        """_check_legacy_layer Detect if the maya scene use Legacy Render Layer or Render Setup"""
        # python command, not working with maya 2018 use mel.eval instead
        # if cmds.mayaHasRenderSetup():
        if mel.eval("mayaHasRenderSetup()"):
            return
        else:
            layers = cmds.listConnections("renderLayerManager")
            if layers[0] != "defaultRenderLayer" or len(layers) > 1:
                for layer in layers:
                    if not layer == "defaultRenderLayer" and cmds.getAttr(layer + ".renderable"):
                        self.worker.log_warning("Legacy Render Layer not supported, result might not be accurate")
                        break

    def _uploaded_layer(self):
        """_uploaded_layer Detect layers in the scene and upload only the active ones"""
        self.uploaded_layers = list()
        render_layers = cmds.listConnections("renderLayerManager")
        for layer in render_layers:
            if not cmds.nodeType(layer) == "renderLayer":
                continue
            # print("node type: ", cmds.nodeType(layer))
            renderable = cmds.getAttr("{}.renderable".format(layer))
            if renderable:
                # print(layer)
                self.uploaded_layers.append(layer)
            else:
                continue
        if not self.uploaded_layers:
            self.worker.log_error("No Render Layer activated, please turn on at least one layer!")

    def _set_renderman_file_format(self):
        if cmds.getAttr("defaultRenderGlobals.currentRenderer") == "renderman":
            file_frmat = cmds.getAttr("rmanGlobals.imageFileFormat")
            if file_frmat:
                if not file_frmat.endswith("<f4>.<ext>"):
                    self.worker.log_warning(
                        "The render output format will be rendered with <scene>_<layer>_<aov>_<f4>.<ext>"
                    )

    def _check_frame_step(self):
        if cmds.getAttr("defaultRenderGlobals.byFrameStep") > 1:
            self.worker.log_warning("Frame step value over 1, this value will be set to 1 in the Ranch farm")

    def _get_disk_cache(self):
        """_get_disk_cache Detect if diskCache nodes exist, log a warning is exist"""
        try:
            disk_caches = cmds.ls(type="diskCache")
            if len(disk_caches) > 1:
                self.worker.log_warning("Cache files not needed, they will be evaluated at render time.")
        except Exception as e:
            pass

    def _check_unique_filename(self):
        """
        Check for unique filename.

        This method checks if the current scene name matches any reference file names.
        If a match is found, it logs an info message indicating that the reference file has the same name as the current scene.

        Parameters:
            None

        Returns:
            None
        """
        # Get the current scene name
        scene_name = os.path.basename(cmds.file(query=True, sceneName=True))
        if not self.JOB_NAME == "":
            current_scene_name = self.JOB_NAME + "." + scene_name.split(".")[-1]
        else:
            current_scene_name = scene_name
        print("Current: ", current_scene_name)

        # Get all reference nodes
        ref_nodes = cmds.ls(type="reference")

        # Iterate through reference nodes
        for ref_node in ref_nodes:
            try:
                if not cmds.referenceQuery(ref_node, isLoaded=True):
                    continue
                # Get the referenced file path
                ref_file_path = cmds.referenceQuery(ref_node, filename=True)
                ref_file_name = os.path.basename(ref_file_path)
                if ref_file_name.endswith("}"):
                    ref_file_name = ref_file_name.split("{")[0]
                # Check if the reference file name matches the current scene name
                if ref_file_name == current_scene_name:
                    self.worker.log_error(
                        "Reference file '{}' has the same name than another scene, rename reference".format(
                            ref_file_name
                        )
                    )
                    self.on_stop()
            except RuntimeError:
                continue

    def _check_optix(self):
        """_check_optix Check if Optix (GPU) is activated - Not compatible for batch render"""
        imagers_list = cmds.ls(et="aiImagerDenoiserOptix")
        if not imagers_list:
            return
        if Utils.get_render_farm()[0] == "CPU":
            if not ARNOLD_VERSION.startswith(("3", "2")):
                for imager in imagers_list:
                    if cmds.getAttr("{}.enable".format(imager)):
                        self.worker.log_error("Optix Denoiser not supported in CPU mode")

    def _check_forbidden_override(self):
        """
        Check if there is a frame range override (forbidden) or resolution override (warning)
        If this override is enabled display error (forbidden) or warning
        """
        render_layers = [
            layer
            for layer in cmds.listConnections("renderLayerManager")
            if cmds.nodeType(layer) == "renderLayer" and cmds.getAttr(layer + ".renderable")
        ]
        # forbidden_override = ["startFrame", "endFrame", "byFrameStep"]
        warning_override = ["width", "height", "startFrame", "endFrame", "byFrameStep"]
        overrides = cmds.ls(type="override")

        for over in overrides:
            if not cmds.listConnections(over + ".parentList"):
                continue
            try:
                global_over = cmds.listConnections(
                    (cmds.listConnections(cmds.listConnections(over + ".parentList")[0] + ".parentList")[0])
                )
                for layer in render_layers:
                    if layer in global_over:
                        layer = layer.replace("rs_", "", 1) if layer[0:3] == "rs_" else layer
                        # if (
                        #     cmds.getAttr(over + ".attribute") in forbidden_override
                        #     and cmds.getAttr(over + ".selfEnabled")
                        #     and cmds.getAttr(over + ".parentEnabled")
                        # ):
                        #     self.worker.log_error(
                        #         "Override "
                        #         + over
                        #         + " on "
                        #         + layer
                        #         + " is forbidden, read Render Layers documentation"
                        #     )
                        #     continue

                        if (
                            cmds.getAttr(over + ".attribute") in warning_override
                            and cmds.getAttr(over + ".selfEnabled")
                            and cmds.getAttr(over + ".parentEnabled")
                        ):
                            self.worker.log_warning(
                                "Override " + over + " on " + layer + " detected, use Ranch option instead"
                            )
            except Exception as e:
                continue

    def quick_check_run(self):
        # à ajouter à la barre de progression
        start_time = time.time()
        self.log_view.clear()
        log_view = 4
        self.set_btns_list_active(log_view)
        self._check_unsupported_engine()
        self._check_rman_xpu()
        self._check_is_light()
        self._check_jobname_long()
        self._check_stillimage()
        self._check_arnold_tx_file()
        self._check_legacy_layer()
        self._uploaded_layer()
        self._set_renderman_file_format()
        self._check_frame_step()
        self._get_disk_cache()
        self._check_unique_filename()
        self._check_optix()
        self._check_forbidden_override()
        print("Quick check in %s seconds" % (time.time() - start_time))


class MainWindow(QMainWindow, Logger, RendererTab, PluginsTab, InfosTab, Quick_check, Archive, Submit):
    JOB_NAME = ""

    def __init__(self):
        pointer = omui.MQtUtil.mainWindow()
        maya_main_window = wrapInstance(int(pointer), QWidget)
        super(MainWindow, self).__init__(maya_main_window)
        Logger.__init__(self)
        RendererTab.__init__(self)
        PluginsTab.__init__(self)
        InfosTab.__init__(self)
        Archive.__init__(self)
        Quick_check.__init__(self)
        Submit.__init__(self)

        self.priority_id = 1
        self.prioritys = list()
        self.__destination = ""
        self.__specific_frames = ""
        self.stopped = False

        self.worker = Worker(ui=self)

        global SUPPORTED_RENDERER
        SUPPORTED_RENDERER = [key["name"] for key in self.get_support_renderer()]

        self.setMinimumWidth(MIN_WIDTH)
        self.setMinimumHeight(MIN_HEIGHT)

        self.__get_priority()
        self.__create_layout()
        self.__createMenuBar()
        self.__create_header()
        self.__create_body()
        self.__create_settings()
        self.create_renderer()
        self.create_plugins()
        self.create_infos()
        self.__create_log()
        self.create_archive()
        self.create_submit()
        self.archive_visibility()
        self.__create_progress_status()
        self.__init_value()
        self.__set_connection()

    def __get_renderer_id(self):
        scene_renderer = str(Utils.get_current_renderer()).capitalize()
        render_device = 0
        renderer_id = 0
        if scene_renderer == "Arnold":
            try:
                render_device = cmds.getAttr("defaultArnoldRenderOptions.renderDevice")
            except Exception as e:
                if Utils.get_maya_version() not in ["2018", "2019"]:
                    QMessageBox.information(
                        self,
                        "RANCHecker",
                        "Render device [GPU, CPU] is not found, please check in render settings",
                    )
            if render_device == 1:
                scene_renderer = scene_renderer + " GPU"
        elif scene_renderer == "Vray":
            render_device = cmds.getAttr("vraySettings.productionEngine")
            scene_renderer = "V-Ray"
            if render_device == 2 or render_device == 3:
                scene_renderer = scene_renderer + " GPU"
        elif scene_renderer == "Renderman":
            scene_renderer = "RenderMan"
        elif scene_renderer == "Mayasoftware":
            scene_renderer = "Maya Software"
        renderer = self.get_support_renderer()
        for rn in renderer:
            if rn["name"] == scene_renderer:
                renderer_id = rn["id"]
                break
        return renderer_id

    def __get_priority(self):
        data = {
            "method": "sweb_get_json",
            "url": "api/renderer-releases/{}/priorities".format(self.__get_renderer_id()),
        }
        resp = self.send_to_ranchsync(data)
        if not resp or resp.get("error"):
            # QMessageBox.warning(self, "RANCHecker", "Could not connect to the ranchcomputing server")
            return False
        for i in resp["farms"]:
            farm = resp["farms"][i]
            for p in reversed(farm):
                self.prioritys.append(p)
        return self.prioritys

    def __init_value(self):
        self.submit_visibility(False)

    def __create_layout(self):
        main_widget = QWidget()
        main_layout = QVBoxLayout()
        main_layout.setSpacing(6)
        main_layout.setContentsMargins(0, 0, 0, 0)
        self.header_layout = QHBoxLayout()
        self.body_layout = QHBoxLayout()
        self.body_layout.setContentsMargins(6, 0, 6, 0)
        self.archive_submit_layout = QHBoxLayout()
        self.archive_submit_layout.setContentsMargins(0, 6, 6, 0)
        self.archive_submit_layout.setSpacing(10)
        self.archive_submit_layout.addStretch()
        self.progress_status_layout = QVBoxLayout()
        main_layout.addLayout(self.header_layout)
        main_layout.addLayout(self.body_layout)
        main_layout.addLayout(self.archive_submit_layout)
        main_layout.addLayout(self.progress_status_layout)
        main_widget.setLayout(main_layout)
        self.setCentralWidget(main_widget)

    def __createMenuBar(self):
        menu_bar = self.menuBar()
        self.guide_action = QAction("Guide", self)
        self.ranchsync_action = QAction("Install RANCHsync", self)
        self.check_update = QAction("Check Update", self)
        help_menu = QMenu("&Help", self)
        ranchsync_menu = QMenu("&RANCHSync", self)
        help_menu.addAction(self.guide_action)
        help_menu.addAction(self.check_update)
        ranchsync_menu.addAction(self.ranchsync_action)
        menu_bar.addMenu(ranchsync_menu)
        menu_bar.addMenu(help_menu)
        self.setMenuBar(menu_bar)

    def __create_header(self):
        self.btn_header = QPushButton()
        try:
            req = Request(self.get_picto_image_url())
            res = urlopen(req)
            image_data = res.read()
            image = QImage()
            image.loadFromData(image_data)
            # Scale the image
            new_width, new_height = 545, 100  # Desired resolution
            scaled_image = image.scaled(new_width, new_height, Qt.KeepAspectRatio, Qt.SmoothTransformation)
            self.btn_header.setIcon(QIcon(QPixmap(scaled_image)))
            if self.is_picto_active():
                self.btn_header.setIconSize(QSize(545, 100))
            else:
                self.btn_header.setIconSize(QSize(545, 80))
            self.btn_header.setStyleSheet("background-color: rgb(29, 29, 53); ")
        except:
            self.btn_header.setText("Ranchcomputing")
        self.header_layout.addWidget(self.btn_header)

    def __create_body(self):
        btns = ["Settings", "Renderer", "Plugins", "Infos", "Log"]
        self.btns_list = QListWidget()
        self.btns_list.setMaximumWidth(80)
        self.stack_view = QStackedWidget()
        self.stack_view.setStyleSheet(
            "QStackedWidget{background-color: rgb(60, 60, 60); border-radius: 5px;\
                                        border: 1px solid rgb(43, 43, 43)}"
        )
        self.btns_list.setStyleSheet("background-color: rgba(0, 0, 0, 0); border-radius: 0px;")
        for btn in btns:
            item = QListWidgetItem()
            item.setText("  " + btn)
            item.setSizeHint(QSize(80, 40))
            self.btns_list.addItem(item)
        item = self.btns_list.item(0)
        self.btns_list.setCurrentItem(item)

        self.body_layout.addWidget(self.btns_list)
        self.body_layout.addWidget(self.stack_view)

    def __create_settings(self):  # Settings tab UI
        settings_widgets = QWidget()
        self.settings_layout = QVBoxLayout()
        self.settings_layout.setContentsMargins(4, 4, 4, 4)
        self.__create_destination_settings()
        self.__create_job_settings()
        # self.__create_asset_group() # TODO do not remove activate after support separate assets
        self.settings_layout.addStretch()
        settings_widgets.setLayout(self.settings_layout)
        self.stack_view.addWidget(settings_widgets)

    def __create_destination_settings(self):
        destination_layout = QHBoxLayout()
        archive_destination = QLabel("Archive destination")
        self.archive_destination_path = QLineEdit()
        self.archive_destination_path.setPlaceholderText(SCENE_PATH)
        self.btn_destination = QPushButton(" ... ")
        self.btn_destination.setMaximumHeight(17)
        destination_layout.addWidget(archive_destination)
        destination_layout.addWidget(self.archive_destination_path)
        destination_layout.addWidget(self.btn_destination)
        self.settings_layout.addLayout(destination_layout)

    def on_tiling_mode_changed(self):
        current_resolution = Utils.get_w_resolution() * Utils.get_h_resolution()
        # self.multibands.setVisible(self.tiling_mode.isChecked())
        self.tiling_update(current_resolution)

    def tiling_update(self, resolution):
        print("Updating Tiling options with resolution: {}".format(resolution))
        current_device = Utils.get_render_farm()[0]
        self.multibands.clear()

        if not self.tiling_mode.isChecked():
            self.multibands.setEnabled(False)
            self.multibands_text.setText("")
            self.multibands_text.setVisible(False)
            return

        if Utils.get_framerange_mode():
            self.multibands.setEnabled(False)
            self.multibands.setVisible(False)
            self.multibands_text.setVisible(True)
            self.multibands_text.setText("Disabled - Animation detected!")
            # self.multibands.clear()
            return

        if RENDERER in ["mayaSoftware"]:
            # self.tiling_mode.stateChanged.emit(self.tiling_mode.setChecked(False))
            self.multibands.setEnabled(False)
            self.multibands.setVisible(False)
            self.multibands_text.setVisible(True)
            self.multibands_text.setText("Not available for {}".format(RENDERER.upper()))
            return

        if current_device == "GPU" and resolution <= 8294399 or current_device == "CPU" and resolution <= 2073599:
            # self.tiling_mode.stateChanged.emit(self.tiling_mode.setChecked(False))
            self.multibands.setEnabled(False)
            self.multibands.setVisible(False)
            self.multibands_text.setVisible(True)
            self.multibands_text.setText("Not available for this resolution")
            return

        #  unsupported settings
        unsupported_dict = {
            "arnold": {
                "format_cmd": (lambda: cmds.getAttr("defaultRenderGlobals.imfPluginKey")),
                "formats": ["maya", "rtoa_denoiser"],
            },
            ## "6:als","23:avi","35:dds","9:eps","0:gif","31:psd","36:psd","12:yuv","22:mov","2:rla","5:sgi","13:sgi","1:pic","63:xpm"
            # "mayaSoftware": {
            #     "format_cmd": (lambda: cmds.getAttr("defaultRenderGlobals.imageFormat")),
            #     "formats": [6, 23, 35, 9, 0, 31, 36, 12, 22, 2, 5, 13, 1, 63],
            # },
            "renderman": {
                "format_cmd": (
                    lambda: re.split("\d+", cmds.listConnections("rmanDefaultDisplay.displayType")[0])[0].split(
                        "d_"
                    )[-1]
                ),
                "formats": ["null", "it", "targa", "texture", "pointcloud"],
            },
            "vray": {
                "format_cmd": (lambda: cmds.getAttr("vraySettings.imageFormatStr")),
                "formats": ["vrimg", "vrst", "sgi"],
            },
        }
        if (
            RENDERER in unsupported_dict.keys()
            and unsupported_dict[RENDERER]["format_cmd"]() in unsupported_dict[RENDERER]["formats"]
        ):
            self.multibands.setEnabled(False)
            self.multibands.setVisible(False)
            self.multibands_text.setVisible(True)
            self.multibands_text.setText(
                "Not supported for {} format".format(str(unsupported_dict[RENDERER]["format_cmd"]()).upper())
            )
            return

        if current_device == "CPU" and resolution >= 2073600:
            print(Utils.get_render_farm()[0])
            self.multibands.setEnabled(False)
            self.multibands.setVisible(False)
            self.multibands_text.setVisible(True)
            self.multibands_text.setText("Auto Tiles")

        else:
            self.multibands.setVisible(True)
            self.multibands.setEnabled(True)
            self.multibands_text.setText("Tiles with MaxStrips")  # to fake create maxstrip
            if 8294400 <= resolution < 33177601:
                self.multibands.addItem("3 Tiles")
            elif 8294400 <= resolution < 132710401:
                self.multibands.addItems(["3 Tiles", "6 Tiles"])
            elif 8294400 <= resolution < 530000001:
                self.multibands.addItems(["3 Tiles", "6 Tiles", "12 Tiles"])
            elif resolution >= 530000000:
                self.multibands.addItems(["6 Tiles", "12 Tiles", "Auto Tiles"])

    def __create_job_settings(self):
        # layout init
        job_group = QGroupBox("Job")
        job_layout = QHBoxLayout()
        email_layout = QHBoxLayout()
        group_layout = QVBoxLayout()
        image_priority_layout = QHBoxLayout()
        # items configuration
        job_name_label = QLabel("Job name")
        self.specify_frames = QCheckBox("Specify frames")
        self.priority_label = QLabel("Priority")
        self.job_name_edit = QLineEdit()
        self.job_frames = QLineEdit()
        self.job_frames.setPlaceholderText("Example: 2,5-10,17-32,99999")
        self.job_frames.setEnabled(False)
        self.job_frames.setStyleSheet("color: #ddd;")
        self.job_frames.setVisible(False)
        self.priority = QComboBox()
        self.tiling_mode = QCheckBox("Use Tiling (Beta)")
        self.tiling_mode.setChecked(False)
        self.tiling_mode.setToolTip(
            "<b>Tiles rendering mode</b>"
            "<br>If you use tiles, your scene will be distributed across as many servers as there are tiles"
            "<br>resulting in faster rendering but at an additional cost"
            "<br><span style='color:darkorange'>Beta: Rendering should be ok but Stitching part may produce an incorrect result, contact us to resolve this"
            "<br><span style='color:green'><b><i>Works for 2k+(CPU), 4K+(GPU) resolutions<i/><b/></span>"
        )
        self.Receive_email = QCheckBox("Receive email")
        self.multibands = QComboBox()
        self.multibands.setEnabled(False)
        self.multibands.setVisible(False)
        self.multibands_text = QLabel("")
        self.multibands_text.setVisible(False)
        self.__set_prioritys()
        # organize items
        job_layout.addWidget(job_name_label)
        job_layout.addWidget(self.job_name_edit)
        job_layout.addWidget(self.priority_label)
        job_layout.addWidget(self.priority)
        email_layout.addWidget(self.specify_frames)
        email_layout.addWidget(self.job_frames)
        email_layout.addWidget(self.Receive_email)
        image_priority_layout.addWidget(self.tiling_mode)
        image_priority_layout.addWidget(self.multibands)
        image_priority_layout.addWidget(self.multibands_text)
        group_layout.addLayout(job_layout)
        group_layout.addLayout(image_priority_layout)
        group_layout.addLayout(email_layout)
        job_group.setLayout(group_layout)
        self.settings_layout.addWidget(job_group)

    def __set_prioritys(self):
        if self.prioritys:
            # fmt: off
            for p in self.prioritys:
                if p["name"].startswith("GPU"):
                    self.priority.addItem(
                        u"{} | {} Cards | Price {}".format(p["name"], p["max_allocated_nodes"], p["ghz_h_price"])
                    )
                else:
                    self.priority.addItem(
                        u"{} | {} Nodes | Price {}".format(
                            p["name"], p["max_allocated_nodes"], p["ghz_h_price"]
                        )
                    )
        # fmt: on
        else:
            self.priority.setVisible(False)
            self.priority_label.setVisible(False)
            self.specify_frames.setVisible(False)
            self.job_frames.setVisible(False)
            self.Receive_email.setVisible(False)

    def __create_asset_group(self):  # FTP UI
        asset_group = QGroupBox("Assets")
        separate_asset_checkbox = QCheckBox("Separate assets from archive")
        asset_layout = QVBoxLayout()
        hostname_pass_layout = QHBoxLayout()
        assets_folder_layout = QHBoxLayout()
        hostname_label = QLabel("Hostname")
        password_label = QLabel("Password")
        use_assets_label = QLabel("Use assets")
        folder_label = QLabel("Folder")
        self.folder_edit = QLineEdit()
        self.assets_combobox = QComboBox()
        self.host_name_edit = QLineEdit()
        self.password_edit = QLineEdit()
        self.btn_connect = QPushButton("Connect")
        hostname_pass_layout.addWidget(hostname_label)
        hostname_pass_layout.addWidget(self.host_name_edit)
        hostname_pass_layout.addWidget(password_label)
        hostname_pass_layout.addWidget(self.password_edit)
        hostname_pass_layout.addWidget(self.btn_connect)
        assets_folder_layout.addWidget(use_assets_label)
        assets_folder_layout.addWidget(self.assets_combobox)
        assets_folder_layout.addWidget(folder_label)
        assets_folder_layout.addWidget(self.folder_edit)
        asset_layout.addWidget(separate_asset_checkbox)
        asset_layout.addLayout(hostname_pass_layout)
        asset_layout.addLayout(assets_folder_layout)
        asset_group.setLayout(asset_layout)
        self.settings_layout.addWidget(asset_group)

    def __create_log(self):
        log_widgets = QWidget()
        log_lyout = QVBoxLayout()
        log_lyout.setContentsMargins(0, 0, 0, 0)
        self.log_view = QListWidget()
        self.log_view.setSpacing(1)
        log_lyout.addWidget(self.log_view)
        log_widgets.setLayout(log_lyout)
        self.stack_view.addWidget(log_widgets)

    def __create_progress_status(self):
        self.status = QLabel()
        self.status.setContentsMargins(6, 0, 0, 0)
        self.progress_bar = QProgressBar()
        self.progress_bar.setMaximumHeight(18)
        self.progress_status_layout.addWidget(self.status)
        self.progress_status_layout.addWidget(self.progress_bar)

    def __set_connection(self):
        self.btn_header.clicked.connect(self.on_header)
        self.btn_archive.clicked.connect(self.on_archive)
        self.btn_destination.clicked.connect(self.on_btn_distination)
        self.btn_stop.clicked.connect(self.on_stop)
        self.btn_submit.clicked.connect(self.on_submit)
        self.btns_list.currentRowChanged.connect(self.on_btns_list)
        self.guide_action.triggered.connect(self.on_guide)
        self.check_update.triggered.connect(self.on_check_update)
        self.ranchsync_action.triggered.connect(self.on_install_ranchsync)
        self.archive_destination_path.textChanged.connect(self.on_destination_changed)
        self.job_name_edit.textChanged.connect(self.on_job_name_changed)
        self.tiling_mode.stateChanged.connect(self.on_tiling_mode_changed)
        self.specify_frames.clicked.connect(self.on_specify_frames)
        self.job_frames.textChanged.connect(self.on_frames_changed)
        self.worker.send_log.connect(self.on_log)
        self.worker.progress_value_text.connect(self.on_progress)
        self.worker.message_emit.connect(self.on_worker_message)
        self.worker.progress_range.connect(self.on_progress_range)
        self.worker.finished.connect(self.on_worker_finished)

    @property
    def job_name(self):
        return self.JOB_NAME

    @job_name.setter
    def job_name(self, name):
        self.JOB_NAME = name

    @property
    def specific_frames(self):
        return self.__specific_frames

    @specific_frames.setter
    def specific_frames(self, range):
        if range:
            self.__specific_frames = {"type": "specific", "specific_frames": range}
        else:
            self.__specific_frames = {}

    @property
    def destination(self):
        return self.__destination

    def get_destination(self):
        if self.destination:
            return self.destination
        return SCENE_PATH

    @destination.setter
    def destination(self, desti):
        self.__destination = desti

    def set_btns_list_active(self, numb):
        log_item = self.btns_list.item(numb)
        self.btns_list.setCurrentItem(log_item)
        self.stack_view.setCurrentIndex(numb)

    def _set_item(self, msg):
        item = QListWidgetItem()
        key = list(msg.keys())[0]
        if key == "WARNING":
            # item.setBackgroundColor(COLOR_WARNING)
            item.setBackground(QBrush(QColor(COLOR_WARNING)))  # PySide6 Fix
        elif key == "INFO":
            # item.setBackgroundColor(COLOR_INFO)
            item.setBackground(QBrush(QColor(COLOR_INFO)))  # PySide6 Fix
        elif key == "ERROR":
            # item.setBackgroundColor(COLOR_ERROR)
            item.setBackground(QBrush(QColor(COLOR_ERROR)))  # PySide6 Fix
        elif key == "SUCCESS":
            # item.setBackgroundColor(COLOR_SUCCESS)
            item.setBackground(QBrush(QColor(COLOR_SUCCESS)))  # PySide6 Fix
        item.setText(msg[key])
        item.setSizeHint(QSize(30, 20))
        # item.setTextColor(QColor(60, 60, 60))
        item.setForeground(QBrush(QColor(60, 60, 60)))  # PySide6 Fix
        return item

    def _set_widget(self, key, value):
        w = QWidget()
        l = QHBoxLayout()
        l.setContentsMargins(0, 0, 0, 0)
        l.setSpacing(0)
        key_lbl = QLabel(key)
        key_lbl.setMaximumWidth(120)
        key_lbl.setStyleSheet("color: #edf3f6; background-color: #5285a6; border-radius: 0px; padding: 4px;")
        value_lbl = QLabel(value)
        value_lbl.setStyleSheet("background-color: #0a3d62; border-radius: 0px; padding: 4px;")
        l.addWidget(key_lbl)
        l.addWidget(value_lbl)
        w.setLayout(l)
        return w

    def on_worker_message(self, msg):
        QMessageBox.information(self, "RANCHecker", msg)

    def on_progress_range(self, progress_range):
        self.progress_bar.setRange(0, progress_range - 1)

    def on_progress(self, text, value):
        self.progress_bar.setValue(value)
        self.status.setText(text)

    def on_worker_finished(self):
        self.archive_visibility(False)
        self.is_running = False
        # if self.stopped or not self.worker._check_log_error():
        if self.stopped:
            self.submit_visibility(False)
            return
        t = int(time.time() - self.worker.archive_creation_time)
        format_t = datetime.timedelta(seconds=t)
        self.worker.log_info("Archive creation time in {}".format(format_t))
        vua_archive_path = os.path.join(self.get_destination(), self.worker.get_archive_name() + ".vua")
        self.worker.log_success("Archive created in [ {} ]".format(vua_archive_path))
        if self.prioritys:
            self.submit_visibility(True)
        else:
            self.submit_visibility(False)

    def on_stop(self):
        self.stopped = True
        self.archive_visibility(self.is_running)
        self.status.setText("")
        self.submit_visibility(False)
        self.is_running = False
        self.worker.on_stop_process()
        self.worker.log_error("Archive creation stopped ..")

    def on_log(self, msg):
        log_view = 4
        self.set_btns_list_active(log_view)
        item = self._set_item(msg)
        self.log_view.addItem(item)

    def on_btn_distination(self):
        destination_path = QFileDialog.getExistingDirectory(self, "Archive Directory")
        self.archive_destination_path.setText(destination_path)

    def on_destination_changed(self, desti):
        self.destination = desti

    def on_job_name_changed(self, name):
        if len(name) >= 21 or UNSUPPORTED_CHARACTER.search(name):
            self.job_name_edit.setStyleSheet("color: red;")
            self.btn_archive.setEnabled(False)
        else:
            self.job_name_edit.setStyleSheet("")
            self.btn_archive.setEnabled(True)
        self.JOB_NAME = name

    def on_frames_changed(self, range):
        supported = "0123456789-,"
        for i in range:
            if i in supported:
                self.job_frames.setStyleSheet("color: #edf3f6;")
            else:
                self.job_frames.setStyleSheet("color: red;")
                break
        self.specific_frames = range

    def on_specify_frames(self):
        # self.job_frames.setEnabled(enabled)
        enabled = self.specify_frames.isChecked()
        self.job_frames.setEnabled(enabled)
        self.job_frames.setVisible(enabled)
        self.job_frames.setText("")
        self.job_frames.setStyleSheet("color: #ddd;")

    def on_install_ranchsync(self):
        webbrowser.open_new_tab("https://www.ranchcomputing.com/fr/download?from=app")

    def on_guide(self):
        webbrowser.open_new_tab("https://doc.ranchcomputing.com/maya:ranchecker-new")

    def on_check_update(self):
        _update = Api()
        _update.check_version()

    def on_btns_list(self, index):
        self.stack_view.setCurrentIndex(index)
        if self.btns_list.item(index).text() == "  Infos":
            self.set_scene_info_item(Utils.get_scene_infos())

        elif self.btns_list.item(index).text() == "  Plugins":
            self.set_plugins_item()

        elif self.btns_list.item(index).text() == "  Renderer":
            self.set_gallery_item()

    def __get_combobox_priority(self):
        prio = self.priority.currentText().split("|")[0]
        for i in self.prioritys:
            if prio.replace(" ", "") == i["name"]:
                self.priority_id = i["id"]
                break

    def on_archive(self):
        self.stopped = False
        self.submit_visibility(False)
        self.quick_check_run()
        self.__get_combobox_priority()
        self._run_process()

    def on_submit(self):
        if self.uploaded_layers:
            for index, layer in enumerate(self.uploaded_layers):
                self.submit(
                    self.__get_renderer_id(),
                    self.priority_id,
                    not self.Receive_email.isChecked(),
                    layer,
                    index,
                )
        else:
            self.submit(
                self.__get_renderer_id(),
                self.priority_id,
                not self.Receive_email.isChecked(),
            )
        self.submit_visibility(False)

    def on_header(self):
        webbrowser.open_new_tab(self.get_picto_target_url())
        # webbrowser.open_new_tab("https://www.ranchcomputing.com/dashboard")

    def closeEvent(self, event):
        self.isClosed = True
        # Any other cleanup code
        event.accept()


class RanchRunCommand(om.MPxCommand):
    COMMAND_NAME = "ranchArchive"

    def __init__(self):
        om.MPxCommand.__init__(self)

    def create_syntax(self):
        syntax = om.MSyntax()

        syntax.addFlag("-d", "-string", om.MSyntax.kString)  # destination
        syntax.addFlag("-j", "-string", om.MSyntax.kString)  # job name
        # syntax.addFlag("-w", "-string", om.MSyntax.kString)  # work space
        return syntax

    def doIt(self, args):
        # command :
        # C:\Maya\Maya2022\bin\maya.exe -batch -file "C:\Maya\CurrentJob\scenes\ranchArchive.mb" -command "ranchArchive;" -log "C:\Maya\maya.log"
        print("start ranchecker ranchArchive command ... : ")
        destination = ""
        job_name = ""
        workspace = "c:\\maya\\currentjob"
        parser = om.MArgParser(self.create_syntax(), args)
        if parser.isFlagSet("-d"):
            destination = parser.flagArgumentString("-d", 0)
            print("destination : {}".format(destination))
        if parser.isFlagSet("-j"):
            job_name = parser.flagArgumentString("-j", 0)
            print("job name : {}".format(job_name))
        wk = Worker(destination=destination, job_name=job_name)
        wk.temp_folder = os.path.abspath(destination)
        print("set temp folder : {}".format(destination))
        if not wk.save_scene():
            raise Exception("coud not save scene")
        print("scene saved ...")

        utils.executeInMainThreadWithResult(wk._create_archive)
        print("RANCHecker archive created in [ {} ]".format(destination + "\\" + job_name + ".vua"))

    @classmethod
    def create(cls):
        return RanchRunCommand()


class RanchCommand(om.MPxCommand):
    COMMAND_NAME = "ranchecker"

    def __init__(self):
        om.MPxCommand.__init__(self)

    def doIt(self, args):
        global RANCH_WINDOW
        try:
            RANCH_WINDOW.close()
            RANCH_WINDOW.deleteLater()
        except:
            pass
        _update = Api()
        _update.check_update()
        RANCH_WINDOW = MainWindow()
        RANCH_WINDOW.setWindowTitle("RANCHecker v" + __version__)
        RANCH_WINDOW.show()

    @classmethod
    def create(cls):
        return RanchCommand()


def add_menu():
    try:
        cmds.deleteUI(cmds.menu("RANCHeckerMenu", l="RANCH", e=1, dai=1, vis=1))
    except:
        pass
    cmds.menu("RANCHeckerMenu", l="RANCH", to=False, p="MayaWindow")
    cmds.menuItem(l="RANCHecker", c="cmds.ranchecker()", i="")
    cmds.refresh()


def remove_menu():
    try:
        cmds.deleteUI(cmds.menu("RANCHeckerMenu", l="RANCH", e=1, dai=1, vis=1))
    except:
        pass


def initializePlugin(plugin):
    author = "ranchcomputing"
    ranch_plugin = om.MFnPlugin(plugin, author, __version__)
    try:
        ranch_plugin.registerCommand(RanchCommand.COMMAND_NAME, RanchCommand.create)
        ranch_plugin.registerCommand(RanchRunCommand.COMMAND_NAME, RanchRunCommand.create)
        add_menu()
    except:
        om.MGlobal.displayError("failed to register command : {0}".format(RanchCommand))


def uninitializePlugin(plugin):
    ranch_plugin = om.MFnPlugin(plugin)
    try:
        RANCH_WINDOW.close()
        ranch_plugin.deregisterCommand(RanchCommand.COMMAND_NAME)
        ranch_plugin.deregisterCommand(RanchRunCommand.COMMAND_NAME)
        remove_menu()
    except:
        om.MGlobal.displayError("failed to deregister command: {0}".format(RanchCommand))
